import React, { useEffect, useCallback } from 'react';
import { m } from 'framer-motion';
import { useToast } from '@/app/components/ToastProvider';
import { useLoader } from '@/shared/providers';
import { api, APIError } from '@/lib/api';
import { getErrorCode } from '@/lib/errorsCodes';
import getDict from '@/lib/dict';
import { checkResponseShape } from '@/lib/checkResponseShape';
import AP from '@/app/types/AP';
import {
  EuiSpacer,
  EuiText,
  EuiFlexGroup,
  EuiFlexItem,
  EuiIcon,
  EuiBadge,
  EuiLoadingSpinner,
} from '@elastic/eui';
import { animations } from '@/features/project-setup/utils';

const dict = getDict('de');

interface LoadingAPProps {
  aps: AP[];
  setAPs: React.Dispatch<React.SetStateAction<AP[]>>;
}

export const LoadingAP: React.FC<LoadingAPProps> = ({ aps, setAPs }) => {
  const { addToast } = useToast();
  const { setLoading } = useLoader();

  const fetchAPs = useCallback(async () => {
    try {
      setLoading(true);
      const data = await api.accessPoints.getAccessPoints();

      // Don't rely on backend, double check response schema
      if (checkResponseShape(data)) {
        setAPs(data);
      } else {
        addToast({
          title: 'Fetch Error',
          color: 'danger',
          text: dict.apFetchToastFail,
        });
      }
      setLoading(false);
    } catch (err) {
      const error = err as APIError;
      const errorCode = getErrorCode('ACCESS_POINT_EXCEPTION', error.status);
      setLoading(false);
      addToast({
        title: errorCode,
        color: 'danger',
        text: dict.apFetchToastFail,
      });
    }
  }, [addToast, setAPs, setLoading]);

  useEffect(() => {
    fetchAPs();
  }, [fetchAPs]);

  return (
    <m.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.6, ease: [0.22, 1, 0.36, 1] }}
      style={{
        position: 'relative',
        zIndex: 1,
      }}
    >
      {/* Access Points Count with Icon */}
      <m.div
        initial={{ scale: 0.8, opacity: 0 }}
        animate={{ scale: 1, opacity: 1 }}
        transition={{
          delay: 0.2,
          duration: 0.5,
          type: 'spring',
          stiffness: 150,
          damping: 15,
        }}
      >
        <EuiFlexGroup
          direction="column"
          alignItems="center"
          gutterSize="m"
          responsive={false}
        >
          {/* Icon Container */}
          <EuiFlexItem grow={false}>
            <m.div
              animate={{
                boxShadow: [
                  '0 4px 20px rgba(0, 191, 179, 0.15)',
                  '0 4px 32px rgba(0, 191, 179, 0.3)',
                  '0 4px 20px rgba(0, 191, 179, 0.15)',
                ],
              }}
              transition={{
                duration: 2.5,
                repeat: Infinity,
                ease: 'easeInOut',
              }}
              style={{
                width: '64px',
                height: '64px',
                borderRadius: '50%',
                background: 'linear-gradient(135deg, #00BFB3 0%, #0077CC 100%)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                position: 'relative',
              }}
            >
              <EuiIcon type="bullseye" size="xl" color="white" />

              {/* Pulse Rings */}
              <m.div
                animate={{
                  scale: [1, 1.5, 1],
                  opacity: [0.6, 0, 0.6],
                }}
                transition={{
                  duration: 2,
                  repeat: Infinity,
                  ease: 'easeOut',
                }}
                style={{
                  position: 'absolute',
                  width: '100%',
                  height: '100%',
                  borderRadius: '50%',
                  border: '2px solid #00BFB3',
                }}
              />
              <m.div
                animate={{
                  scale: [1, 1.5, 1],
                  opacity: [0.4, 0, 0.4],
                }}
                transition={{
                  duration: 2,
                  repeat: Infinity,
                  ease: 'easeOut',
                  delay: 0.5,
                }}
                style={{
                  position: 'absolute',
                  width: '100%',
                  height: '100%',
                  borderRadius: '50%',
                  border: '2px solid #00BFB3',
                }}
              />
            </m.div>
          </EuiFlexItem>

          {/* Count Badge */}
          <EuiFlexItem grow={false}>
            <m.div
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{
                delay: 0.4,
                duration: 0.6,
                type: 'spring',
                stiffness: 200,
                damping: 15,
              }}
            >
              <EuiBadge
                color="success"
                style={{
                  fontSize: '20px',
                  fontWeight: 700,
                  padding: '12px 24px',
                  background: 'linear-gradient(135deg, #00BFB3 0%, #00a89d 100%)',
                  color: 'white',
                  border: 'none',
                  boxShadow: '0 4px 12px rgba(0, 191, 179, 0.3)',
                }}
              >
                <EuiFlexGroup
                  gutterSize="s"
                  alignItems="center"
                  responsive={false}
                >
                  <EuiFlexItem grow={false}>
                    <EuiIcon type="check" color="white" />
                  </EuiFlexItem>
                  <EuiFlexItem grow={false}>
                    {aps.length} {dict.detected}
                  </EuiFlexItem>
                </EuiFlexGroup>
              </EuiBadge>
            </m.div>
          </EuiFlexItem>
        </EuiFlexGroup>
      </m.div>

      <EuiSpacer size="l" />

      {/* Auto-redirect Message */}
      <m.div
        initial={{ opacity: 0, y: 10 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.6, duration: 0.5 }}
        style={{
          textAlign: 'center',
          background: '#F5F7FA',
          borderRadius: '8px',
          padding: '16px 24px',
          border: '1px solid #e0e5ea',
        }}
      >
        <EuiFlexGroup
          direction="column"
          alignItems="center"
          gutterSize="s"
          responsive={false}
        >
          <EuiFlexItem grow={false}>
            <EuiFlexGroup
              gutterSize="s"
              alignItems="center"
              responsive={false}
            >
              <EuiFlexItem grow={false}>
                <EuiLoadingSpinner size="m" />
              </EuiFlexItem>
              <EuiFlexItem grow={false}>
                <EuiText
                  size="m"
                  style={{
                    color: '#69707d',
                    fontWeight: 500,
                  }}
                >
                  {dict.autoRedirect}
                </EuiText>
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiFlexItem>

          {/* Progress Indicator Dots */}
          <EuiFlexItem grow={false}>
            <EuiFlexGroup
              gutterSize="xs"
              alignItems="center"
              responsive={false}
            >
              {[0, 1, 2].map((i) => (
                <EuiFlexItem grow={false} key={i}>
                  <m.div
                    animate={{
                      scale: [1, 1.3, 1],
                      opacity: [0.3, 1, 0.3],
                    }}
                    transition={{
                      duration: 1.5,
                      repeat: Infinity,
                      ease: 'easeInOut',
                      delay: i * 0.2,
                    }}
                    style={{
                      width: '6px',
                      height: '6px',
                      borderRadius: '50%',
                      background: '#00BFB3',
                    }}
                  />
                </EuiFlexItem>
              ))}
            </EuiFlexGroup>
          </EuiFlexItem>
        </EuiFlexGroup>
      </m.div>
    </m.div>
  );
};
