import React, { useState, useEffect, useMemo } from 'react';
import { m } from 'framer-motion';
import { animations } from '@/features/project-setup/utils';
import { useIsSmallScreen } from '@/features/project-setup/hooks/useMediaQuery';
import getDict from '@/lib/dict';
import {
  EuiLoadingSpinner,
  EuiPanel,
  EuiSpacer,
  EuiText,
  EuiTitle,
  EuiIcon,
} from '@elastic/eui';

const dict = getDict('de');

interface LoadingScreenProps {
  text: string;
}

export const LoadingScreen: React.FC<LoadingScreenProps> = ({ text }) => {
  const [progress, setProgress] = useState(0);
  const isSmall = useIsSmallScreen();

  useEffect(() => {
    const interval = setInterval(() => {
      setProgress((prev) => {
        if (prev >= 100) {
          clearInterval(interval);
          return 100;
        }
        return prev + Math.random() * 8 + 2;
      });
    }, 150);

    return () => clearInterval(interval);
  }, []);

  const iconStyle = useMemo(
    () => ({
      width: isSmall ? '80px' : '100px',
      height: isSmall ? '80px' : '100px',
      margin: '0 auto 24px',
      background: 'white',
      borderRadius: '50%',
      display: 'flex',
      alignItems: 'center',
      justifyContent: 'center',
      border: '3px solid #e0e5ea',
      position: 'relative' as const,
      zIndex: 1,
    }),
    [isSmall]
  );

  const progressPercentage = Math.round(progress);

  return (
    <m.div
      key="loading-screen"
      variants={animations.pageVariants}
      initial="initial"
      animate="in"
      exit="out"
      style={{
        maxWidth: '640px',
        margin: '0 auto',
        padding: isSmall ? '16px' : '32px',
      }}
    >
      {/* Main Loading Panel */}
      <m.div
        initial={{ opacity: 0, y: 40 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
      >
        <EuiPanel
          hasBorder
          hasShadow={false}
          paddingSize="none"
          style={{
            background: 'linear-gradient(to bottom, #ffffff 0%, #f8f9fa 100%)',
            borderRadius: '20px',
            padding: isSmall ? '40px 24px' : '56px 48px',
            textAlign: 'center',
            position: 'relative',
            overflow: 'hidden',
            border: '1px solid #e0e5ea',
          }}
        >
          {/* Animated Background Circles */}
          <m.div
            animate={{
              scale: [1, 1.2, 1],
              opacity: [0.03, 0.05, 0.03],
            }}
            transition={{
              duration: 8,
              repeat: Infinity,
              ease: 'easeInOut',
            }}
            style={{
              position: 'absolute',
              top: '-50%',
              right: '-20%',
              width: '500px',
              height: '500px',
              borderRadius: '50%',
              background: 'radial-gradient(circle, #00BFB3 0%, transparent 70%)',
            }}
          />
          <m.div
            animate={{
              scale: [1, 1.3, 1],
              opacity: [0.02, 0.04, 0.02],
            }}
            transition={{
              duration: 10,
              repeat: Infinity,
              ease: 'easeInOut',
              delay: 1,
            }}
            style={{
              position: 'absolute',
              bottom: '-30%',
              left: '-10%',
              width: '400px',
              height: '400px',
              borderRadius: '50%',
              background: 'radial-gradient(circle, #0077CC 0%, transparent 70%)',
            }}
          />

          {/* Animated Icon with Pulse */}
          <m.div
            initial={{ opacity: 0, scale: 0.5 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{
              delay: 0.2,
              duration: 0.8,
              type: 'spring',
              stiffness: 120,
              damping: 12,
            }}
            style={{ position: 'relative', zIndex: 1 }}
          >
            <m.div
              animate={{
                boxShadow: [
                  '0 8px 32px rgba(0, 191, 179, 0.2)',
                  '0 8px 48px rgba(0, 191, 179, 0.35)',
                  '0 8px 32px rgba(0, 191, 179, 0.2)',
                ],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: 'easeInOut',
              }}
              style={iconStyle}
            >
              <m.div
                animate={{ rotate: 360 }}
                transition={{
                  duration: 3,
                  repeat: Infinity,
                  ease: 'linear',
                }}
              >
                <EuiIcon
                  type="online"
                  size="xxl"
                  style={{ color: '#00BFB3', fontSize: isSmall ? '40px' : '48px' }}
                />
              </m.div>
            </m.div>
          </m.div>

          {/* Title with Fade Up */}
          <m.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4, duration: 0.7, ease: [0.22, 1, 0.36, 1] }}
            style={{ position: 'relative', zIndex: 1 }}
          >
            <EuiTitle size={isSmall ? 'm' : 'l'}>
              <h1
                style={{
                  color: '#1a1c21',
                  margin: 0,
                  fontWeight: 600,
                  letterSpacing: '-0.02em',
                }}
              >
                {text}
              </h1>
            </EuiTitle>
            <EuiSpacer size="s" />
            <m.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 0.6, duration: 0.7 }}
            >
              <EuiText size="m" style={{ color: '#69707d', fontWeight: 400 }}>
                {dict.wait}...
              </EuiText>
            </m.div>
          </m.div>

          <EuiSpacer size="xl" />

          {/* Progress Bar Container */}
          <m.div
            initial={{ opacity: 0, scale: 0.95 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: 0.8, duration: 0.5 }}
            style={{
              position: 'relative',
              zIndex: 1,
              maxWidth: '400px',
              margin: '0 auto',
            }}
          >
            <div
              style={{
                position: 'relative',
                background: '#e0e5ea',
                borderRadius: '12px',
                height: '12px',
                overflow: 'hidden',
              }}
            >
              <m.div
                initial={{ width: '0%' }}
                animate={{ width: `${progress}%` }}
                transition={{ duration: 0.3, ease: 'easeOut' }}
                style={{
                  height: '100%',
                  background: 'linear-gradient(90deg, #00BFB3 0%, #0077CC 100%)',
                  borderRadius: '12px',
                  position: 'relative',
                }}
              >
                {/* Shimmer Effect */}
                <m.div
                  animate={{
                    x: ['-100%', '100%'],
                  }}
                  transition={{
                    duration: 1.5,
                    repeat: Infinity,
                    ease: 'easeInOut',
                  }}
                  style={{
                    position: 'absolute',
                    top: 0,
                    left: 0,
                    width: '100%',
                    height: '100%',
                    background:
                      'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.3) 50%, transparent 100%)',
                  }}
                />
              </m.div>
            </div>

            <EuiSpacer size="m" />

            {/* Percentage and Spinner */}
            <m.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 1, duration: 0.5 }}
              style={{
                display: 'flex',
                justifyContent: 'center',
                alignItems: 'center',
                gap: '12px',
              }}
            >
              <EuiLoadingSpinner size="m" />
              <EuiText
                size="m"
                style={{
                  fontWeight: 600,
                  color: '#00BFB3',
                  fontSize: isSmall ? '16px' : '18px',
                  letterSpacing: '-0.01em',
                }}
              >
                {progressPercentage}%
              </EuiText>
            </m.div>
          </m.div>

          {/* Decorative Bottom Line */}
          <m.div
            initial={{ scaleX: 0 }}
            animate={{ scaleX: 1 }}
            transition={{ delay: 1.2, duration: 0.8, ease: 'easeOut' }}
            style={{
              position: 'absolute',
              bottom: 0,
              left: '50%',
              transform: 'translateX(-50%)',
              width: '60%',
              height: '3px',
              background: 'linear-gradient(90deg, transparent 0%, #00BFB3 50%, transparent 100%)',
              borderRadius: '3px',
            }}
          />
        </EuiPanel>
      </m.div>

      {/* Footer Info */}
      <m.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ delay: 1.4, duration: 0.5 }}
        style={{ marginTop: '24px', textAlign: 'center' }}
      >
        <EuiText size="xs" color="subdued">
          <p>
            <EuiIcon type="clock" size="s" style={{ marginRight: '6px' }} />
            Dies kann einige Momente dauern
          </p>
        </EuiText>
      </m.div>
    </m.div>
  );
};
