/**
 * LoadingScreens Component
 * Handles project modal, module loading, error, and complete screens
 */

import React, { useEffect } from 'react';
import {
  ProjectSetupScreen,
  ModuleLoadingScreen,
  CompleteScreen,
} from '@/features/project-setup/components/screens';
import { Error } from '@/features/project-setup/components/ui';
import { SCREEN_TYPES } from '@/features/project-setup/utils/constants';
import { TIMING, PROGRESS } from '@/lib/uiConstants';
import { processPdfFromFile } from '@/features/pdf-editor/utils/pdfUtils';
import { handleError } from '@/lib/errorHandler';
import { AccessPointMarkerPoint } from '@/features/access-points';

import {DeviceDetails, PDFFile, Toast, ViewportAction} from "@/shared/types";

interface LoadFirstProjectResult {
  project?: {
    accessPoints: AccessPointMarkerPoint[];
    currentPage?: number;
    pageRotation?: number;
    viewportState?: {
      pageViewports?: Record<number, { zoomLevel: number; panOffset: { x: number; y: number } }>;
    };
    pdfFile: {
      uploadedPath: string;
    };
  };
  pdfBlob?: Blob;
  pdfFileName?: string;
}

interface LoadingScreensProps {
  currentScreen: string;
  progress: number;
  loadingError: string | null;
  goToStep: string;
  projectName: string;
  saved: boolean;
  navigateTo: (screen: string) => void;
  setProgress: (progress: number | ((prev: number) => number)) => void;
  setgoToStep: (step: string) => void;
  resetSaved: () => void;
  setProjectName: (name: string) => void;
  saveProject: () => Promise<void>;
  loadFirstProject: () => Promise<LoadFirstProjectResult>;
  loadProjectsFromApi: () => Promise<void>;
  generateDeviceDetails: (type: string) => DeviceDetails;
  setAccessPoints: (points: AccessPointMarkerPoint[]) => void;
  setCurrentPage: (page: number) => void;
  setPageViewports: (viewports: Record<number, { zoomLevel: number; panOffset: { x: number; y: number } }>) => void;
  setPdfPages: (pages: string[]) => void;
  setPdfFile: (file: PDFFile) => void;
  setProjectFound: (found: boolean) => void;
  setStatus: (status: boolean) => void;
  dispatch: React.Dispatch<ViewportAction>;
  addToast: (toast: Toast) => void;
  setLoadingError: (error: string | null) => void;
}

export function LoadingScreens({
  currentScreen,
  progress,
  loadingError,
  goToStep,
  projectName,
  saved,
  navigateTo,
  setProgress,
  setgoToStep,
  resetSaved,
  setProjectName,
  saveProject,
  loadFirstProject,
  loadProjectsFromApi,
  generateDeviceDetails,
  setAccessPoints,
  setCurrentPage,
  setPageViewports,
  setPdfPages,
  setPdfFile,
  setProjectFound,
  setStatus,
  dispatch,
  addToast,
  setLoadingError
}: LoadingScreensProps) {
  // Loading progress for module loading screen
  useEffect(() => {
    if (currentScreen === SCREEN_TYPES.MODULE_LOADING) {
      const interval = setInterval(() => {
        if (!saved) {
          setProgress(prev => {
            if (prev >= PROGRESS.MAX) {
              clearInterval(interval);
              setTimeout(() => {
                setgoToStep(SCREEN_TYPES.PROJECT_MODAL);
              }, TIMING.COMPLETE_SCREEN_DELAY);
              return PROGRESS.MAX;
            }
            return prev + Math.random() * PROGRESS.RANDOM_MAX + PROGRESS.RANDOM_MIN;
          });
        } else {
          setProgress(prev => {
            if (prev >= PROGRESS.MAX) {
              clearInterval(interval);
              setTimeout(() => {
                navigateTo(SCREEN_TYPES.COMPLETE);
                resetSaved();
              }, TIMING.COMPLETE_SCREEN_DELAY);
              return PROGRESS.MAX;
            }
            return prev + Math.random() * PROGRESS.RANDOM_MAX + PROGRESS.RANDOM_MIN;
          });
        }
      }, TIMING.PROGRESS_INTERVAL);
      return () => clearInterval(interval);
    }
  }, [currentScreen, saved, navigateTo, setProgress, setgoToStep, resetSaved]);

  // Reload project when complete
  const handleReloadProject = async () => {
    try {
      setStatus(true);
      const result = await loadFirstProject();

      if (result?.project) {
        await loadProjectsFromApi();

        if (result.project.pageRotation) {
          dispatch({ type: 'SET_ROTATION', payload: result.project.pageRotation });
        }

        const accessPointsWithDetails = result.project.accessPoints.map((ap: AccessPointMarkerPoint) => ({
          ...ap,
          deviceDetails: generateDeviceDetails(ap.type)
        }));

        setAccessPoints(accessPointsWithDetails);
        setCurrentPage(result.project.currentPage || 0);

        if (result.project.viewportState?.pageViewports) {
          setPageViewports(result.project.viewportState.pageViewports);
        }

        if (result.pdfBlob && result.pdfFileName) {
          const pdfFileWithPath = Object.assign(
            new File([result.pdfBlob], result.pdfFileName, {
              type: 'application/pdf'
            }),
            { uploadedPath: result.project.pdfFile.uploadedPath }
          ) as PDFFile;

          const pages = await processPdfFromFile(pdfFileWithPath);
          setPdfPages(pages);
          setPdfFile(pdfFileWithPath);
          setProjectFound(true);
        }
      }

      setStatus(false);
    } catch (error) {
      setStatus(false);
      handleError(error, { addToast, showToast: true, setLoadingError });
    }

    navigateTo(SCREEN_TYPES.PROJECT_SETUP);
  };

  return (
    <>
      {/* Project Setup Modal */}
      {currentScreen === SCREEN_TYPES.PROJECT_MODAL && (
        <ProjectSetupScreen
          projectName={projectName}
          setProjectName={setProjectName}
          onContinue={async () => {
            await saveProject();
            navigateTo(SCREEN_TYPES.MODULE_LOADING);
          }}
        />
      )}

      {/* Module Loading Screen */}
      {currentScreen === SCREEN_TYPES.MODULE_LOADING && (
        <ModuleLoadingScreen progress={progress} />
      )}

      {/* Error Screen */}
      {currentScreen === SCREEN_TYPES.FAIL && (
        <Error loadingError={loadingError} step={goToStep} />
      )}

      {/* Complete Screen */}
      {currentScreen === SCREEN_TYPES.COMPLETE && (
        <CompleteScreen onFinish={handleReloadProject} />
      )}
    </>
  );
}
