import React, { useState, useEffect, useRef, useCallback, useImperativeHandle, forwardRef } from 'react';
import { m, animations } from 'framer-motion';
import {
  EuiPanel,
  EuiTitle,
  EuiText,
  EuiFlexItem,
  EuiSpacer,
  EuiFormRow,
  EuiFieldText,
  EuiLoadingSpinner,
  EuiButton,
  EuiFlexGroup,
  EuiCallOut,
  EuiIcon,
  EuiBadge,
  EuiCard,
  EuiButtonEmpty,
} from '@elastic/eui';
import getDict from '@/lib/dict';
import { useLoader } from '@/shared/providers';
import { getErrorCode } from '@/lib/errorsCodes';
import { useToast } from '@/app/components/ToastProvider';
import { api, APIError } from '@/lib/api';
import type { Network } from '@/lib/api/endpoints/networks';
import { useIsSmallScreen } from '@/features/project-setup/hooks/useMediaQuery';

const dict = getDict('de');

interface NetworkSetupScreenProps {
  networkName: string;
  setNetworkName: (name: string) => void;
  onContinue: () => void;
  onSkip: () => void;
  onError?: (setter: (error: string | null) => void) => void;
}

export const NetworkSetupScreen: React.FC<NetworkSetupScreenProps> = ({
  networkName,
  setNetworkName,
  onContinue,
  onSkip,
  onError,
}) => {
  const [networks, setNetworks] = useState<Network[]>([]);
  const [isFetchingNetworks, setIsFetchingNetworks] = useState(true);
  const [isProcessing, setIsProcessing] = useState(false);
  const [creationError, setCreationError] = useState<string | null>(null);
  const { setLoading, loading } = useLoader();
  const { addToast } = useToast();
  const inputRef = useRef<HTMLInputElement>(null);
  const isSmall = useIsSmallScreen();

  // Call onError callback with the error setter function when the component mounts
  useEffect(() => {
    if (onError) {
      onError(setCreationError);
    }
  }, [onError]);

  const fetchNetworks = useCallback(async () => {
    try {
      setIsFetchingNetworks(true);
      const data = await api.networks.getNetworks();
      setNetworks(data);
    } catch (err) {
      const error = err as APIError;
      const errorCode = getErrorCode('NETWORK_EXCEPTION', error.status);
      addToast({
        title: errorCode,
        color: 'danger',
        text: dict.networkFetchToastFail,
      });
    } finally {
      setIsFetchingNetworks(false);
    }
  }, [addToast]);

  useEffect(() => {
    fetchNetworks();
  }, []);

  // Restore focus to the input field when action loading completes
  useEffect(() => {
    if (!loading && inputRef.current) {
      inputRef.current.focus();
    }
  }, [loading]);

  // Reset processing state when loading completes
  useEffect(() => {
    if (!loading && isProcessing) {
      setIsProcessing(false);
    }
  }, [loading, isProcessing]);

  // Track previous network name to detect changes
  const prevNetworkNameRef = useRef(networkName);

  // Clear error only when user changes the network name (not just when it exists)
  useEffect(() => {
    if (creationError && networkName !== prevNetworkNameRef.current) {
      setCreationError(null);
    }
    prevNetworkNameRef.current = networkName;
  }, [networkName, creationError]);

  const handleContinue = useCallback(() => {
    setCreationError(null);
    setIsProcessing(true);
    setLoading(true);
    onContinue();
  }, [onContinue, setLoading]);

  const handleSkip = useCallback(() => {
    setCreationError(null);
    setIsProcessing(true);
    setLoading(true);
    onSkip();
  }, [onSkip, setLoading]);

  const isCreateDisabled = loading || !networkName.trim();
  const isSkipDisabled = loading || (networks.length === 0 && !networkName.trim());

  const headerTitle = networks.length === 0
    ? dict.noNetwork
    : `${dict.network} ${dict.create}`;

  const headerSubtitle = networks.length === 0
    ? dict.networkInfo
    : `${dict.networkInfo} oder ${dict.skip}`;

  const networkCountText = networks.length > 0
    ? `${networks.length} ${dict.network}${networks.length > 1 ? '(en)' : ''} ${dict.found}`
    : '';

  const createButtonLabel = dict.create.charAt(0).toUpperCase() + dict.create.slice(1);

  return (
    <m.div
      key="network-setup-screen"
      variants={animations.pageVariants}
      initial="initial"
      animate="in"
      exit="out"
      style={{
        maxWidth: '680px',
        margin: '0 auto',
        padding: isSmall ? '16px' : '24px',
      }}
    >
      {/* Header with Icon */}
      <m.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.1, duration: 0.4 }}
        style={{ textAlign: 'center', marginBottom: '32px' }}
      >
        <EuiFlexGroup
          direction="column"
          alignItems="center"
          gutterSize="m"
          responsive={false}
        >
          <EuiFlexItem grow={false}>
            <div
              style={{
                width: '64px',
                height: '64px',
                borderRadius: '50%',
                background: 'linear-gradient(135deg, #00BFB3 0%, #0077CC 100%)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                boxShadow: '0 4px 12px rgba(0, 191, 179, 0.2)',
              }}
            >
              <EuiIcon type="globe" size="xl" color="white" />
            </div>
          </EuiFlexItem>

          <EuiFlexItem grow={false}>
            <EuiTitle size="l">
              <h1 style={{ margin: 0, fontWeight: 600 }}>{headerTitle}</h1>
            </EuiTitle>
          </EuiFlexItem>

          {networks.length > 0 && (
            <EuiFlexItem grow={false}>
              <EuiBadge color="success" iconType="check">
                {networkCountText}
              </EuiBadge>
            </EuiFlexItem>
          )}

          <EuiFlexItem grow={false}>
            <EuiText
              color="subdued"
              textAlign="center"
              style={{ maxWidth: '480px' }}
            >
              <p>{headerSubtitle}</p>
            </EuiText>
          </EuiFlexItem>
        </EuiFlexGroup>
      </m.div>

      {/* Main Card */}
      <m.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.2, duration: 0.4 }}
      >
        <EuiPanel
          hasBorder
          hasShadow={false}
          paddingSize="none"
          style={{
            border: '1px solid #D3DAE6',
            borderRadius: '8px',
            overflow: 'hidden',
          }}
        >
          {/* Info CallOut */}
          {networks.length === 0 && (
            <EuiCallOut
              size="m"
              color="primary"
              iconType="iInCircle"
              title="Netzwerk erforderlich"
              style={{ borderRadius: 0, border: 'none', borderBottom: '1px solid #D3DAE6' }}
            >
              <p>
                Erstellen Sie ein Netzwerk, um mit der Konfiguration Ihrer Access Points zu
                beginnen.
              </p>
            </EuiCallOut>
          )}

          {/* Form Section */}
          <div style={{ padding: '32px' }}>
            {/* Error CallOut */}
            {creationError && (
              <>
                <EuiCallOut
                  title="Fehler beim Erstellen"
                  color="danger"
                  iconType="alert"
                  size="m"
                >
                  <p>{creationError}</p>
                  <EuiSpacer size="s" />
                  <EuiText size="s" color="subdued">
                    Bitte überprüfen Sie Ihre Eingaben und versuchen Sie es erneut.
                  </EuiText>
                </EuiCallOut>
                <EuiSpacer size="m" />
              </>
            )}

            <EuiFormRow
              label={
                <span style={{ fontSize: '14px', fontWeight: 500 }}>
                  <EuiIcon type="tag" size="m" style={{ marginRight: '8px' }} />
                  {dict.name}
                </span>
              }
              fullWidth
              helpText="Geben Sie einen eindeutigen Namen für Ihr Netzwerk ein"
              isInvalid={!!creationError}
            >
              <EuiFieldText
                value={networkName}
                onChange={(e) => setNetworkName(e.target.value)}
                placeholder="z.B. Büro-Netzwerk oder Hauptstandort"
                fullWidth
                disabled={loading}
                autoFocus
                inputRef={inputRef}
                icon="globe"
                style={{ fontSize: '16px' }}
                isInvalid={!!creationError}
              />
            </EuiFormRow>

            <EuiSpacer size="xl" />

            {/* Action Buttons */}
            <EuiFlexGroup
              direction={isSmall ? 'column' : 'row'}
              gutterSize="m"
              alignItems="center"
              justifyContent="center"
            >
              {isProcessing && (
                <EuiFlexItem grow={false}>
                  <EuiFlexGroup
                    direction="column"
                    alignItems="center"
                    gutterSize="s"
                    responsive={false}
                  >
                    <EuiFlexItem grow={false}>
                      <EuiLoadingSpinner size="l" />
                    </EuiFlexItem>
                    <EuiFlexItem grow={false}>
                      <EuiText size="s" color="subdued">
                        Verarbeitung läuft...
                      </EuiText>
                    </EuiFlexItem>
                  </EuiFlexGroup>
                </EuiFlexItem>
              )}

              <EuiFlexItem grow={isSmall}>
                <EuiButton
                  fill
                  color="primary"
                  onClick={handleContinue}
                  disabled={isCreateDisabled}
                  fullWidth={isSmall}
                  size="m"
                  iconType="check"
                  style={{ minWidth: isSmall ? 'auto' : '160px' }}
                >
                  {createButtonLabel}
                </EuiButton>
              </EuiFlexItem>

              <EuiFlexItem grow={isSmall}>
                <EuiButtonEmpty
                  onClick={handleSkip}
                  disabled={isSkipDisabled}
                  fullWidth={isSmall}
                  size="m"
                  iconType="arrowRight"
                  iconSide="right"
                  style={{ minWidth: isSmall ? 'auto' : '160px' }}
                >
                  {dict.skip}
                </EuiButtonEmpty>
              </EuiFlexItem>
            </EuiFlexGroup>
          </div>
        </EuiPanel>
      </m.div>

      {/* Footer Help Text */}
      <m.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ delay: 0.4, duration: 0.4 }}
        style={{ marginTop: '24px', textAlign: 'center' }}
      >
        <EuiText size="xs" color="subdued">
          <p>
            <EuiIcon type="questionInCircle" size="s" style={{ marginRight: '6px' }} />
            Sie können weitere Netzwerke später in den Einstellungen hinzufügen
          </p>
        </EuiText>
      </m.div>
    </m.div>
  );
};
