import React, { useRef, useEffect } from 'react';
import { m } from 'framer-motion';
import {
  EuiPanel,
  EuiTitle,
  EuiButton,
  EuiText,
  EuiSpacer,
  EuiFormRow,
  EuiFieldText,
  EuiIcon,
  EuiFlexGroup,
  EuiFlexItem,
  EuiCallOut,
} from '@elastic/eui';
import { animations } from '@/features/project-setup/utils';
import { useIsSmallScreen } from '@/features/project-setup/hooks/useMediaQuery';
import getDict from '@/lib/dict';

const dict = getDict('de');

interface ProjectSetupScreenProps {
  projectName: string;
  setProjectName: (name: string) => void;
  onContinue: () => void;
}

export const ProjectSetupScreen: React.FC<ProjectSetupScreenProps> = ({
  projectName,
  setProjectName,
  onContinue,
}) => {
  const inputRef = useRef<HTMLInputElement>(null);
  const isSmall = useIsSmallScreen();

  useEffect(() => {
    // Auto-focus the input field when component mounts
    if (inputRef.current) {
      inputRef.current.focus();
    }
  }, []);

  const handleKeyPress = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter' && projectName.trim()) {
      onContinue();
    }
  };

  const isDisabled = !projectName.trim();

  return (
    <m.div
      key="project-setup-screen"
      variants={animations.pageVariants}
      initial="initial"
      animate="in"
      exit="out"
      style={{
        maxWidth: '680px',
        margin: '0 auto',
        padding: isSmall ? '16px' : '24px',
      }}
    >
      {/* Header with Icon */}
      <m.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.1, duration: 0.4 }}
        style={{ textAlign: 'center', marginBottom: '32px' }}
      >
        <EuiFlexGroup
          direction="column"
          alignItems="center"
          gutterSize="m"
          responsive={false}
        >
          <EuiFlexItem grow={false}>
            <m.div
              initial={{ scale: 0.5, opacity: 0, rotate: -180 }}
              animate={{ scale: 1, opacity: 1, rotate: 0 }}
              transition={{
                delay: 0.2,
                duration: 0.8,
                type: 'spring',
                stiffness: 120,
                damping: 12,
              }}
            >
              <m.div
                animate={{
                  boxShadow: [
                    '0 4px 24px rgba(0, 191, 179, 0.2)',
                    '0 4px 40px rgba(0, 191, 179, 0.35)',
                    '0 4px 24px rgba(0, 191, 179, 0.2)',
                  ],
                }}
                transition={{
                  duration: 3,
                  repeat: Infinity,
                  ease: 'easeInOut',
                }}
                style={{
                  width: '72px',
                  height: '72px',
                  borderRadius: '50%',
                  background: 'linear-gradient(135deg, #00BFB3 0%, #0077CC 100%)',
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                  boxShadow: '0 4px 24px rgba(0, 191, 179, 0.2)',
                }}
              >
                <EuiIcon type="documentEdit" size="xl" color="white" />
              </m.div>
            </m.div>
          </EuiFlexItem>

          <EuiFlexItem grow={false}>
            <m.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.3, duration: 0.5 }}
            >
              <EuiTitle size="l">
                <h1 style={{ margin: 0, fontWeight: 600 }}>{dict.finalStep}</h1>
              </EuiTitle>
            </m.div>
          </EuiFlexItem>

          <EuiFlexItem grow={false}>
            <m.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 0.4, duration: 0.5 }}
            >
              <EuiText
                color="subdued"
                textAlign="center"
                style={{ maxWidth: '480px' }}
              >
                <p>Geben Sie Ihrem Projekt einen eindeutigen Namen</p>
              </EuiText>
            </m.div>
          </EuiFlexItem>
        </EuiFlexGroup>
      </m.div>

      {/* Main Card */}
      <m.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.2, duration: 0.4 }}
      >
        <EuiPanel
          hasBorder
          hasShadow={false}
          paddingSize="none"
          style={{
            background: 'linear-gradient(to bottom, #ffffff 0%, #f8f9fa 100%)',
            border: '1px solid #D3DAE6',
            borderRadius: '12px',
            overflow: 'hidden',
            position: 'relative',
          }}
        >
          {/* Subtle Background Decoration */}
          <m.div
            animate={{
              scale: [1, 1.1, 1],
              opacity: [0.02, 0.04, 0.02],
            }}
            transition={{
              duration: 8,
              repeat: Infinity,
              ease: 'easeInOut',
            }}
            style={{
              position: 'absolute',
              top: '-30%',
              right: '-20%',
              width: '300px',
              height: '300px',
              borderRadius: '50%',
              background: 'radial-gradient(circle, #00BFB3 0%, transparent 70%)',
              pointerEvents: 'none',
            }}
          />

          {/* Info CallOut */}
          <EuiCallOut
            size="m"
            color="primary"
            iconType="iInCircle"
            title="Letzter Schritt"
            style={{
              borderRadius: 0,
              border: 'none',
              borderBottom: '1px solid #D3DAE6',
            }}
          >
            <p>
              Wählen Sie einen aussagekräftigen Namen für Ihr Projekt. Dieser Name
              hilft Ihnen, Ihre Projekte später leichter zu identifizieren.
            </p>
          </EuiCallOut>

          {/* Form Section */}
          <div style={{ padding: '32px', position: 'relative', zIndex: 1 }}>
            <m.div
              initial={{ opacity: 0, y: 10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.4, duration: 0.4 }}
            >
              <EuiFormRow
                label={
                  <span style={{ fontSize: '14px', fontWeight: 500 }}>
                    <EuiIcon type="tag" size="m" style={{ marginRight: '8px' }} />
                    {dict.projectNameText}
                  </span>
                }
                fullWidth
                helpText="z.B. Bürogebäude-Nord, Campus-Hauptgebäude, oder Produktionshalle-A"
              >
                <EuiFieldText
                  value={projectName}
                  onChange={(e) => setProjectName(e.target.value)}
                  onKeyPress={handleKeyPress}
                  placeholder="Geben Sie einen Projektnamen ein..."
                  fullWidth
                  autoFocus
                  inputRef={inputRef}
                  icon="document"
                  style={{ fontSize: '16px' }}
                  prepend={
                    <EuiIcon
                      type="folderOpen"
                      style={{ padding: '0 8px', color: '#69707d' }}
                    />
                  }
                />
              </EuiFormRow>

              <EuiSpacer size="xl" />

              {/* Action Button */}
              <EuiFlexGroup
                direction="column"
                alignItems="center"
                gutterSize="m"
              >
                <EuiFlexItem grow={false} style={{ width: isSmall ? '100%' : 'auto' }}>
                  <m.div
                    initial={{ opacity: 0, scale: 0.95 }}
                    animate={{ opacity: 1, scale: 1 }}
                    transition={{ delay: 0.6, duration: 0.4 }}
                    style={{ width: '100%' }}
                  >
                    <EuiButton
                      fill
                      color="primary"
                      onClick={onContinue}
                      disabled={isDisabled}
                      fullWidth={isSmall}
                      size="m"
                      iconType="arrowRight"
                      iconSide="right"
                      style={{
                        minWidth: isSmall ? 'auto' : '200px',
                        background: isDisabled
                          ? undefined
                          : 'linear-gradient(135deg, #00BFB3 0%, #00a89d 100%)',
                      }}
                    >
                      {dict.continue}
                    </EuiButton>
                  </m.div>
                </EuiFlexItem>

                {/* Hint Text */}
                <EuiFlexItem grow={false}>
                  <m.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ delay: 0.8, duration: 0.4 }}
                  >
                    <EuiText size="xs" color="subdued" textAlign="center">
                      <p>
                        <EuiIcon type="returnKey" size="s" style={{ marginRight: '4px' }} />
                        Drücken Sie Enter zum Fortfahren
                      </p>
                    </EuiText>
                  </m.div>
                </EuiFlexItem>
              </EuiFlexGroup>
            </m.div>
          </div>

          {/* Decorative Bottom Line */}
          <m.div
            initial={{ scaleX: 0 }}
            animate={{ scaleX: 1 }}
            transition={{ delay: 0.6, duration: 0.8, ease: 'easeOut' }}
            style={{
              position: 'absolute',
              bottom: 0,
              left: '50%',
              transform: 'translateX(-50%)',
              width: '60%',
              height: '3px',
              background: 'linear-gradient(90deg, transparent 0%, #00BFB3 50%, transparent 100%)',
              borderRadius: '3px',
            }}
          />
        </EuiPanel>
      </m.div>

      {/* Footer Help Text */}
      <m.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ delay: 0.8, duration: 0.4 }}
        style={{ marginTop: '24px', textAlign: 'center' }}
      >
        <EuiText size="xs" color="subdued">
          <p>
            <EuiIcon type="iInCircle" size="s" style={{ marginRight: '6px' }} />
            Der Projektname kann später in den Einstellungen geändert werden
          </p>
        </EuiText>
      </m.div>
    </m.div>
  );
};
