import React, { useEffect, useState } from 'react';
import Image from 'next/image';
import { m } from 'framer-motion';
import {
  EuiFlexItem,
  EuiFlexGroup,
  EuiPanel,
  EuiTitle,
  EuiSpacer,
  EuiText,
  EuiButton,
  EuiLoadingSpinner,
  EuiIcon,
  EuiCallOut,
} from '@elastic/eui';
import { animations } from '@/features/project-setup/utils/animations';
import { useIsSmallScreen } from '@/features/project-setup/hooks/useMediaQuery';
import branding from '@/lib/branding';
import getDict from '@/lib/dict';
import { useLoader } from '@/shared/providers';
import { SCREEN_TYPES } from '@/features/project-setup/utils/constants';
import { useScreenManager } from '@/features/project-setup/hooks/useScreenManager';

const dict = getDict('de');

interface WelcomeScreenProps {
  onNext: () => void;
  status: any;
  savedProjects: any[];
}

export const WelcomeScreen: React.FC<WelcomeScreenProps> = ({ onNext, status, savedProjects }) => {
  const { completeCurrentStep } = useLoader();
  const { navigateTo } = useScreenManager();
  const [isLoading, setIsLoading] = useState(true);
  const isSmall = useIsSmallScreen();

  // Auto-navigate if projects exist
  useEffect(() => {
    if (savedProjects.length > 0) {
      const timer = setTimeout(() => {
        completeCurrentStep();
        navigateTo(SCREEN_TYPES.PROJECT_SETUP);
      }, 4000);
      return () => clearTimeout(timer);
    }
  }, [savedProjects, completeCurrentStep, navigateTo]);

  // Simulated loading delay
  useEffect(() => {
    const timer = setTimeout(() => {
      setIsLoading(false);
    }, 2000);
    return () => clearTimeout(timer);
  }, []);

  const hasProjects = savedProjects.length > 0;

  return (
    <m.div
      key="welcome-screen"
      variants={animations.pageVariants}
      initial="initial"
      animate="in"
      exit="out"
      style={{
        maxWidth: '720px',
        margin: '0 auto',
        padding: isSmall ? '16px' : '32px',
      }}
    >
      {/* Hero Section */}
      <m.div
        initial={{ opacity: 0, y: 40 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.8, ease: [0.22, 1, 0.36, 1] }}
      >
        <EuiPanel
          hasBorder
          hasShadow={false}
          paddingSize="none"
          style={{
            background: 'linear-gradient(to bottom, #ffffff 0%, #f8f9fa 100%)',
            borderRadius: '20px',
            padding: isSmall ? '40px 24px' : '56px 48px',
            textAlign: 'center',
            position: 'relative',
            overflow: 'hidden',
            border: '1px solid #e0e5ea',
          }}
        >
          {/* Animated Background Circles */}
          <m.div
            animate={{
              scale: [1, 1.2, 1],
              opacity: [0.03, 0.05, 0.03],
            }}
            transition={{
              duration: 8,
              repeat: Infinity,
              ease: 'easeInOut',
            }}
            style={{
              position: 'absolute',
              top: '-50%',
              right: '-20%',
              width: '600px',
              height: '600px',
              borderRadius: '50%',
              background: 'radial-gradient(circle, #00BFB3 0%, transparent 70%)',
            }}
          />
          <m.div
            animate={{
              scale: [1, 1.3, 1],
              opacity: [0.02, 0.04, 0.02],
            }}
            transition={{
              duration: 10,
              repeat: Infinity,
              ease: 'easeInOut',
              delay: 1,
            }}
            style={{
              position: 'absolute',
              bottom: '-30%',
              left: '-10%',
              width: '500px',
              height: '500px',
              borderRadius: '50%',
              background: 'radial-gradient(circle, #0077CC 0%, transparent 70%)',
            }}
          />

          {/* Logo with Pulse Animation */}
          <m.div
            initial={{ opacity: 0, scale: 0.5, rotate: -180 }}
            animate={{ opacity: 1, scale: 1, rotate: 0 }}
            transition={{
              delay: 0.2,
              duration: 0.8,
              type: 'spring',
              stiffness: 120,
              damping: 12,
            }}
            style={{ position: 'relative', zIndex: 1 }}
          >
            <m.div
              animate={{
                boxShadow: [
                  '0 8px 32px rgba(0, 191, 179, 0.2)',
                  '0 8px 48px rgba(0, 191, 179, 0.35)',
                  '0 8px 32px rgba(0, 191, 179, 0.2)',
                ],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: 'easeInOut',
              }}
              style={{
                width: isSmall ? '100px' : '120px',
                height: isSmall ? '100px' : '120px',
                margin: '0 auto 32px',
                background: 'white',
                borderRadius: '50%',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                border: '3px solid #e0e5ea',
              }}
            >
              <Image
                src="/logo.svg"
                alt="Logo"
                width={isSmall ? 60 : 70}
                height={isSmall ? 60 : 70}
                priority
              />
            </m.div>
          </m.div>

          {/* Title with Fade Up */}
          <m.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4, duration: 0.7, ease: [0.22, 1, 0.36, 1] }}
            style={{ position: 'relative', zIndex: 1 }}
          >
            <EuiTitle size={isSmall ? 'l' : 'xl'}>
              <h1
                style={{
                  color: '#1a1c21',
                  margin: 0,
                  fontWeight: 700,
                  letterSpacing: '-0.02em',
                }}
              >
                {dict.welcome} {branding.productName}
              </h1>
            </EuiTitle>
            <EuiSpacer size="m" />
            <m.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 0.6, duration: 0.7 }}
            >
              <EuiText size="m" style={{ color: '#69707d', fontWeight: 400 }}>
                Professionelle Konfiguration & Verwaltung von Access Points
              </EuiText>
            </m.div>
          </m.div>

          {/* Decorative Bottom Line */}
          <m.div
            initial={{ scaleX: 0 }}
            animate={{ scaleX: 1 }}
            transition={{ delay: 0.8, duration: 0.8, ease: 'easeOut' }}
            style={{
              position: 'absolute',
              bottom: 0,
              left: '50%',
              transform: 'translateX(-50%)',
              width: '60%',
              height: '3px',
              background: 'linear-gradient(90deg, transparent 0%, #00BFB3 50%, transparent 100%)',
              borderRadius: '3px',
            }}
          />
        </EuiPanel>
      </m.div>

      <EuiSpacer size="xl" />

      {/* Content Section */}
      <m.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.4, duration: 0.5 }}
      >
        <EuiPanel
          hasBorder
          hasShadow={false}
          paddingSize="l"
          style={{
            borderRadius: '12px',
            border: '1px solid #D3DAE6',
          }}
        >
          {/* Loading State */}
          {isLoading && (
            <m.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              transition={{ duration: 0.3 }}
              style={{ textAlign: 'center', padding: '32px 0' }}
            >
              <EuiLoadingSpinner size="xl" />
              <EuiSpacer size="m" />
              <EuiText color="subdued" size="m">
                <p>{dict.mapLoad}</p>
              </EuiText>
            </m.div>
          )}

          {/* Results State */}
          {!isLoading && (
            <m.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.4 }}
            >
              {/* Status CallOut */}
              {hasProjects ? (
                <EuiCallOut
                  title="Projekte gefunden"
                  color="success"
                  iconType="check"
                  size="m"
                >
                  <p>
                    <strong>{savedProjects.length}</strong>{' '}
                    {savedProjects.length === 1 ? 'Projekt wurde' : 'Projekte wurden'} gefunden.
                    Sie werden automatisch zur Projektauswahl weitergeleitet...
                  </p>
                  <EuiSpacer size="s" />
                  <EuiFlexGroup gutterSize="xs" alignItems="center">
                    <EuiFlexItem grow={false}>
                      <EuiLoadingSpinner size="m" />
                    </EuiFlexItem>
                    <EuiFlexItem grow={false}>
                      <EuiText size="s" color="subdued">
                        Weiterleitung in wenigen Sekunden...
                      </EuiText>
                    </EuiFlexItem>
                  </EuiFlexGroup>
                </EuiCallOut>
              ) : (
                <>
                  <div style={{ textAlign: 'center' }}>
                    <div
                      style={{
                        width: '80px',
                        height: '80px',
                        margin: '0 auto 24px',
                        background: '#F5F7FA',
                        borderRadius: '50%',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                      }}
                    >
                      <EuiIcon type="folderOpen" size="xxl" color="subdued" />
                    </div>

                    <EuiTitle size="m">
                      <h2 style={{ margin: 0 }}>{dict.noProject}</h2>
                    </EuiTitle>

                    <EuiSpacer size="m" />

                    <EuiText color="subdued" size="m" textAlign="center" style={{ maxWidth: '480px', margin: '0 auto' }}>
                      <p>{dict.setup}</p>
                    </EuiText>

                    <EuiSpacer size="xl" />

                    {/* Feature Cards */}
                    <EuiFlexGroup
                      direction={isSmall ? 'column' : 'row'}
                      gutterSize="m"
                      responsive={false}
                      style={{ marginBottom: '32px' }}
                    >
                      <EuiFlexItem>
                        <div
                          style={{
                            padding: '16px',
                            borderRadius: '8px',
                            background: '#F5F7FA',
                            textAlign: 'center',
                          }}
                        >
                          <EuiIcon type="globe" size="l" color="primary" />
                          <EuiSpacer size="s" />
                          <EuiText size="s">
                            <strong>Netzwerk konfigurieren</strong>
                          </EuiText>
                        </div>
                      </EuiFlexItem>

                      <EuiFlexItem>
                        <div
                          style={{
                            padding: '16px',
                            borderRadius: '8px',
                            background: '#F5F7FA',
                            textAlign: 'center',
                          }}
                        >
                          <EuiIcon type="bullseye" size="l" color="primary" />
                          <EuiSpacer size="s" />
                          <EuiText size="s">
                            <strong>Access Points platzieren</strong>
                          </EuiText>
                        </div>
                      </EuiFlexItem>

                      <EuiFlexItem>
                        <div
                          style={{
                            padding: '16px',
                            borderRadius: '8px',
                            background: '#F5F7FA',
                            textAlign: 'center',
                          }}
                        >
                          <EuiIcon type="save" size="l" color="primary" />
                          <EuiSpacer size="s" />
                          <EuiText size="s">
                            <strong>Projekt speichern</strong>
                          </EuiText>
                        </div>
                      </EuiFlexItem>
                    </EuiFlexGroup>

                    {/* CTA Button */}
                    <EuiButton
                      fill
                      color="primary"
                      size="m"
                      onClick={onNext}
                      iconType="arrowRight"
                      iconSide="right"
                      style={{ minWidth: isSmall ? '100%' : '200px' }}
                    >
                      {dict.getStarted}
                    </EuiButton>
                  </div>
                </>
              )}
            </m.div>
          )}
        </EuiPanel>
      </m.div>

      {/* Footer */}
      <m.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ delay: 0.6, duration: 0.5 }}
        style={{ marginTop: '24px', textAlign: 'center' }}
      >
        <EuiText size="xs" color="subdued">
          <p>
            <EuiIcon type="questionInCircle" size="s" style={{ marginRight: '6px' }} />
            Bei Fragen oder Problemen kontaktieren Sie den Support
          </p>
        </EuiText>
      </m.div>
    </m.div>
  );
};
