import React, { useState } from 'react';
import { EuiToolTip, EuiPopover } from '@elastic/eui';
import { NODE, SPACING, Z_INDEX, COLORS, POSITION, DIMENSIONS, SHADOWS } from '@/lib/uiConstants';

export const AccessPointLayer = React.memo(({ accessPointTypes, rotation, currentPageAccessPoints, editingNodeId, allowNodeMovement, zoomLevel, handleAccessPointDragStart, startEditingNode, showNodeDetails, handleTouchStartAccessPoint }) => {
  const [contextMenuNodeId, setContextMenuNodeId] = useState(null);

  return <div style={{
    position: 'absolute',
    inset: 0,
    pointerEvents: 'none',
    transform: `rotate(${rotation}deg)`,
    transformOrigin: 'center center'
  }}>
    {

      currentPageAccessPoints.map((ap) => {
        const accessPoint = accessPointTypes.find(t => t.id === ap.type);
        const isEditing = editingNodeId === ap.id;

        return (
          <div
            key={ap.id}
            style={{
              position: 'absolute',
              left: `${ap.x}px`,
              top: `${ap.y}px`,
              transform: `translate(-50%, -50%) scale(${1 / zoomLevel})`,
              pointerEvents: 'auto',
              cursor: isEditing ? 'default' : (allowNodeMovement ? 'grab' : 'pointer'),
              zIndex: Z_INDEX.ACCESS_POINT,
              touchAction: isEditing ? 'auto' : 'none'
            }}
            draggable={!isEditing}
            onDragStart={(e) => !isEditing && handleAccessPointDragStart(e, ap)}
            onTouchStart={(e) => !isEditing && handleTouchStartAccessPoint(e, ap)}
            onMouseDown={(e) => e.stopPropagation()}
          >
            <EuiPopover
              id={`contextMenu-${ap.id}`}
              isOpen={contextMenuNodeId === ap.id}
              closePopover={() => setContextMenuNodeId(null)}
              button={
                <EuiToolTip
                  content={
                    isEditing ? "Editing name..." : (
                      <div>
                        <div style={{ fontWeight: 'bold', marginBottom: `${SPACING.MARGIN_SMALL}px` }}>{ap.name}</div>
                      </div>
                    )
                  }
                  position="top"
                >
                  <div

                    onContextMenu={(e) => {
                      e.preventDefault();
                      setContextMenuNodeId(ap.id);
                    }}
                    onDoubleClick={() => startEditingNode(ap)}
                    onClick={() => !isEditing && showNodeDetails(ap)}
                  >
                    <div style={{ display: 'flex', alignItems: 'center', gap: `${SPACING.GAP_MEDIUM}px` }}>
                      {/* Mini crosshair preview */}
                      <div style={{ position: 'relative', width: `${NODE.SIZE}px`, height: `${NODE.SIZE}px`, flexShrink: 0 }}>
                        {/* Outer circle */}
                        <div
                          style={{
                            width: `${NODE.SIZE}px`,
                            height: `${NODE.SIZE}px`,
                            backgroundColor: accessPoint?.online ? COLORS.ONLINE : COLORS.OFFLINE,
                            borderRadius: '50%',
                            border: `${NODE.BORDER_WIDTH}px solid white`,
                            boxShadow: SHADOWS.LIGHT
                          }}
                        />
                        {/* Inner white circle */}
                        <div
                          style={{
                            position: 'absolute',
                            top: `${POSITION.NODE_INNER_TOP}px`,
                            left: `${POSITION.NODE_INNER_LEFT}px`,
                            width: `${NODE.INNER_SIZE}px`,
                            height: `${NODE.INNER_SIZE}px`,
                            backgroundColor: 'white',
                            borderRadius: '50%'
                          }}
                        />
                        {/* Center dot */}
                        <div
                          style={{
                            position: 'absolute',
                            top: `${POSITION.NODE_CENTER_TOP}px`,
                            left: `${POSITION.NODE_CENTER_LEFT}px`,
                            width: `${NODE.CENTER_SIZE}px`,
                            height: `${NODE.CENTER_SIZE}px`,
                            backgroundColor: accessPoint?.online ? COLORS.ONLINE : COLORS.OFFLINE,
                            borderRadius: '50%'
                          }}
                        />
                        {/* Mini crosshairs */}
                        <div
                          style={{
                            position: 'absolute',
                            top: `${POSITION.CROSSHAIR_HORIZONTAL_TOP}px`,
                            left: `${POSITION.CROSSHAIR_HORIZONTAL_LEFT}px`,
                            width: `${NODE.SIZE}px`,
                            height: `${DIMENSIONS.CROSSHAIR_LINE_HEIGHT}px`,
                            backgroundColor: 'none',
                            opacity: 0.8
                          }}
                        />
                        <div
                          style={{
                            position: 'absolute',
                            top: `${POSITION.CROSSHAIR_VERTICAL_TOP}px`,
                            left: `${POSITION.CROSSHAIR_VERTICAL_LEFT}px`,
                            width: `${DIMENSIONS.CROSSHAIR_LINE_WIDTH}px`,
                            height: `${NODE.SIZE}px`,
                            backgroundColor: 'none',
                            opacity: 0.8
                          }}
                        />
                      </div>
                    </div>
                  </div>
                </EuiToolTip>
              }
              panelPaddingSize="none"
              anchorPosition="downCenter"
            >
            </EuiPopover>

          </div>
        );
      })}
  </div>
}, (prevProps, nextProps) => {
  // Custom comparison for performance
  return (
    prevProps.rotation === nextProps.rotation &&
        prevProps.editingNodeId === nextProps.editingNodeId &&
        prevProps.allowNodeMovement === nextProps.allowNodeMovement &&
        prevProps.zoomLevel === nextProps.zoomLevel &&
        prevProps.currentPageAccessPoints.length === nextProps.currentPageAccessPoints.length &&
        prevProps.currentPageAccessPoints.every((ap, i) =>
          ap.id === nextProps.currentPageAccessPoints[i]?.id &&
            ap.x === nextProps.currentPageAccessPoints[i]?.x &&
            ap.y === nextProps.currentPageAccessPoints[i]?.y
        )
  );
});

AccessPointLayer.displayName = 'AccessPointLayer';
