import React from 'react';
import { m } from 'framer-motion';
import {
  EuiPanel,
  EuiFlexGroup,
  EuiFlexItem,
  EuiText,
  EuiIcon,
  EuiBadge,
  useEuiTheme,
} from '@elastic/eui';
import { useLoader } from '@/shared/providers';

interface Step {
  label: string;
  description?: string;
}

interface StepProgressProps {
  steps?: Step[] | number;
  currentStep?: number;
  completedSteps?: number[] | boolean[];
  size?: 'sm' | 'md' | 'lg';
  showLabels?: boolean;
  showPercentage?: boolean;
  className?: string;
  style?: React.CSSProperties;
}

const StepProgress: React.FC<StepProgressProps> = ({
  steps = [],
  currentStep = 0,
  completedSteps = [],
  size = 'md',
  showLabels = true,
  showPercentage = false,
  className = '',
  style = {},
}) => {
  const { euiTheme } = useEuiTheme();

  // Handle both array of step objects and simple number
  const stepCount = Array.isArray(steps) ? steps.length : steps;
  const stepLabels = Array.isArray(steps) ? steps : Array.from({ length: steps }, (_, i) => ({ label: `Schritt ${i + 1}` }));

  const getCompletedStatus = (stepIndex: number) => {
    if (Array.isArray(completedSteps)) {
      if (typeof completedSteps[0] === 'boolean') {
        return completedSteps[stepIndex] || false;
      } else {
        return completedSteps.includes(stepIndex);
      }
    }
    return false;
  };

  // Size configurations matching EUI design system
  const sizeConfig = {
    sm: {
      stepSize: 28,
      iconSize: 's' as const,
      fontSize: '12px',
      lineHeight: 2,
    },
    md: {
      stepSize: 36,
      iconSize: 'm' as const,
      fontSize: '14px',
      lineHeight: 3,
    },
    lg: {
      stepSize: 44,
      iconSize: 'l' as const,
      fontSize: '16px',
      lineHeight: 4,
    },
  };

  const config = sizeConfig[size];
  const percentage = Math.round(((currentStep + 1) / stepCount) * 100);

  return (
    <div className={className} style={{ width: '100%', ...style }}>
      {/* Header with percentage */}
      {showPercentage && (
        <EuiFlexGroup justifyContent="spaceBetween" alignItems="center" gutterSize="none" style={{ marginBottom: euiTheme.size.m }}>
          <EuiFlexItem grow={false}>
            <EuiText size="s" color="subdued">
              <strong>Fortschritt</strong>
            </EuiText>
          </EuiFlexItem>
          <EuiFlexItem grow={false}>
            <EuiBadge color="primary">{percentage}%</EuiBadge>
          </EuiFlexItem>
        </EuiFlexGroup>
      )}

      <EuiPanel
        hasBorder
        hasShadow={false}
        paddingSize="m"
        style={{
          border: `1px solid ${euiTheme.colors.lightShade}`,
          borderRadius: euiTheme.border.radius.medium,
          background: euiTheme.colors.emptyShade,
        }}
      >
        <EuiFlexGroup
          direction="row"
          alignItems="center"
          gutterSize="none"
          responsive={false}
          style={{ position: 'relative' }}
        >
          {stepLabels.map((step: Step, index: number) => {
            const isCompleted = getCompletedStatus(index);
            const isCurrent = index === currentStep;
            const isPending = index > currentStep;

            // Determine step status color using EUI colors
            const getStepColor = () => {
              if (isCompleted) return euiTheme.colors.success;
              if (isCurrent) return euiTheme.colors.primary;
              return euiTheme.colors.lightShade;
            };

            const stepColor = getStepColor();

            return (
              <React.Fragment key={index}>
                {/* Step indicator */}
                <EuiFlexItem grow={false}>
                  <m.div
                    initial={{ scale: 0.8, opacity: 0 }}
                    animate={{
                      scale: isCurrent ? 1.05 : 1,
                      opacity: 1,
                    }}
                    transition={{
                      delay: index * 0.1,
                      duration: 0.4,
                      ease: [0.4, 0, 0.2, 1],
                    }}
                    style={{
                      display: 'flex',
                      flexDirection: 'column',
                      alignItems: 'center',
                      gap: euiTheme.size.xs,
                      minWidth: showLabels ? '100px' : 'auto',
                      textAlign: 'center',
                    }}
                  >
                    {/* Step circle */}
                    <m.div
                      whileHover={{ scale: 1.1 }}
                      style={{
                        width: config.stepSize,
                        height: config.stepSize,
                        borderRadius: '50%',
                        backgroundColor: stepColor,
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        border: isCurrent ? `3px solid ${euiTheme.colors.primary}` : `2px solid ${stepColor}`,
                        boxShadow: isCurrent
                          ? `0 4px 12px ${euiTheme.colors.primary}40`
                          : isCompleted
                            ? `0 2px 8px ${euiTheme.colors.success}30`
                            : 'none',
                        transition: 'all 0.3s ease',
                        position: 'relative',
                        zIndex: 2,
                      }}
                    >
                      {isCompleted ? (
                        <EuiIcon type="check" size={config.iconSize} color="white" />
                      ) : (
                        <EuiText
                          size="s"
                          style={{
                            color: isCurrent ? 'white' : euiTheme.colors.subduedText,
                            fontWeight: isCurrent ? 700 : 600,
                            fontSize: config.fontSize,
                          }}
                        >
                          {index + 1}
                        </EuiText>
                      )}

                      {/* Pulse animation for current step */}
                      {isCurrent && (
                        <m.div
                          animate={{
                            scale: [1, 1.3, 1],
                            opacity: [0.5, 0, 0.5],
                          }}
                          transition={{
                            duration: 2,
                            repeat: Infinity,
                            ease: 'easeInOut',
                          }}
                          style={{
                            position: 'absolute',
                            inset: -4,
                            borderRadius: '50%',
                            border: `2px solid ${euiTheme.colors.primary}`,
                            zIndex: -1,
                          }}
                        />
                      )}
                    </m.div>

                    {/* Step label */}
                    {showLabels && step.label && (
                      <EuiText
                        size="xs"
                        textAlign="center"
                        style={{
                          fontWeight: isCurrent ? 600 : 400,
                          color: isPending ? euiTheme.colors.subduedText : euiTheme.colors.text,
                          lineHeight: 1.3,
                          maxWidth: '90px',
                          opacity: isPending ? 0.6 : 1,
                        }}
                      >
                        {step.label}
                      </EuiText>
                    )}

                    {/* Step description (optional) */}
                    {showLabels && step.description && (
                      <EuiText
                        size="xs"
                        textAlign="center"
                        color="subdued"
                        style={{
                          lineHeight: 1.2,
                          maxWidth: '100px',
                          fontSize: '11px',
                        }}
                      >
                        {step.description}
                      </EuiText>
                    )}
                  </m.div>
                </EuiFlexItem>

                {/* Connecting line between steps */}
                {index < stepCount - 1 && (
                  <EuiFlexItem style={{ position: 'relative', minWidth: '40px' }}>
                    {/* Background line */}
                    <div
                      style={{
                        position: 'absolute',
                        top: showLabels ? config.stepSize / 2 : '50%',
                        left: 0,
                        right: 0,
                        height: config.lineHeight,
                        backgroundColor: euiTheme.colors.lightShade,
                        borderRadius: euiTheme.border.radius.small,
                        transform: 'translateY(-50%)',
                      }}
                    />

                    {/* Animated progress line */}
                    <m.div
                      initial={{ scaleX: 0 }}
                      animate={{
                        scaleX: isCompleted ? 1 : 0,
                      }}
                      transition={{
                        delay: index * 0.15,
                        duration: 0.6,
                        ease: [0.4, 0, 0.2, 1],
                      }}
                      style={{
                        position: 'absolute',
                        top: showLabels ? config.stepSize / 2 : '50%',
                        left: 0,
                        right: 0,
                        height: config.lineHeight,
                        backgroundColor: euiTheme.colors.success,
                        borderRadius: euiTheme.border.radius.small,
                        transform: 'translateY(-50%)',
                        transformOrigin: 'left',
                        boxShadow: isCompleted ? `0 2px 6px ${euiTheme.colors.success}40` : 'none',
                      }}
                    />
                  </EuiFlexItem>
                )}
              </React.Fragment>
            );
          })}
        </EuiFlexGroup>
      </EuiPanel>
    </div>
  );
};

// Step configurations
export const stepConfigs = {
  numbered: [
    { label: 'Willkommen', description: 'Start' },
    { label: 'Netzwerk', description: 'Konfiguration' },
    { label: 'Projekt', description: 'Setup' },
    { label: 'Access Points', description: 'Platzierung' },
    { label: 'Abschluss', description: 'Fertig' },
  ],
};

// Demo Component with professional EUI styling
export const MultiStepDemo: React.FC = () => {
  const { currentStep, completedSteps } = useLoader();
  const { euiTheme } = useEuiTheme();

  // Debug logging
  React.useEffect(() => {
    console.log('[MultiStepDemo] currentStep:', currentStep, 'completedSteps:', completedSteps);
  }, [currentStep, completedSteps]);

  return (
    <m.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.4 }}
      style={{
        maxWidth: '800px',
        margin: '0 auto',
      }}
    >
      <EuiPanel
        hasBorder
        hasShadow={false}
        paddingSize="l"
        style={{
          border: `1px solid ${euiTheme.colors.lightShade}`,
          borderRadius: euiTheme.border.radius.medium,
          background: 'white',
        }}
      >
        {/* Header */}
        <EuiFlexGroup direction="column" gutterSize="m" style={{ marginBottom: euiTheme.size.l }}>
          <EuiFlexItem>
            <EuiFlexGroup alignItems="center" gutterSize="s">
              <EuiFlexItem grow={false}>
                <EuiIcon type="sortUp" size="l" color="primary" />
              </EuiFlexItem>
              <EuiFlexItem>
                <EuiText size="m">
                  <h3 style={{ margin: 0, fontWeight: 600 }}>Setup-Fortschritt</h3>
                </EuiText>
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiFlexItem>

          <EuiFlexItem>
            <EuiText size="s" color="subdued">
              Schritt {currentStep + 1} von {stepConfigs.numbered.length}
            </EuiText>
          </EuiFlexItem>
        </EuiFlexGroup>

        {/* Step Progress */}
        <StepProgress
          steps={stepConfigs.numbered}
          currentStep={currentStep}
          completedSteps={completedSteps}
          size="md"
          showLabels={true}
          showPercentage={true}
        />
      </EuiPanel>
    </m.div>
  );
};

export default StepProgress;