import { useState, useEffect } from 'react';

export const useMediaQuery = (query: string): boolean => {
  const [matches, setMatches] = useState(false);

  useEffect(() => {
    // SSR guard
    if (typeof window === 'undefined') return;

    const media = window.matchMedia(query);

    // Set initial value
    setMatches(media.matches);

    // Create listener for changes
    const listener = (e: MediaQueryListEvent) => {
      setMatches(e.matches);
    };

    // Add listener
    media.addEventListener('change', listener);

    // Cleanup
    return () => media.removeEventListener('change', listener);
  }, [query]);

  return matches;
};

/**
 * Predefined breakpoints matching common screen sizes
 */
export const BREAKPOINTS = {
  xs: '(max-width: 480px)',
  sm: '(max-width: 640px)',
  md: '(max-width: 768px)',
  lg: '(max-width: 1024px)',
  xl: '(max-width: 1280px)',
  xxl: '(max-width: 1536px)',
} as const;

/**
 * useIsSmallScreen - Hook to detect small screens (≤640px)
 *
 * Replaces the old isSmallScreen() function
 *
 * @example
 * const isSmall = useIsSmallScreen();
 * return <div>{isSmall ? <MobileLayout /> : <DesktopLayout />}</div>
 */
export const useIsSmallScreen = (): boolean => {
  return useMediaQuery(BREAKPOINTS.sm);
};

/**
 * useIsMediumScreen - Hook to detect medium screens (≤768px)
 */
export const useIsMediumScreen = (): boolean => {
  return useMediaQuery(BREAKPOINTS.md);
};

/**
 * useIsLargeScreen - Hook to detect large screens (≤1024px)
 */
export const useIsLargeScreen = (): boolean => {
  return useMediaQuery(BREAKPOINTS.lg);
};

/**
 * useIsMobile - Detects if device is likely mobile (touch + small screen)
 */
export const useIsMobile = (): boolean => {
  const isSmall = useIsSmallScreen();
  const [hasTouch, setHasTouch] = useState(false);

  useEffect(() => {
    if (typeof window === 'undefined') return;
    setHasTouch('ontouchstart' in window || navigator.maxTouchPoints > 0);
  }, []);

  return isSmall && hasTouch;
};

/**
 * usePrefersDarkMode - Detects if user prefers dark color scheme
 */
export const usePrefersDarkMode = (): boolean => {
  return useMediaQuery('(prefers-color-scheme: dark)');
};

/**
 * usePrefersReducedMotion - Detects if user prefers reduced motion
 */
export const usePrefersReducedMotion = (): boolean => {
  return useMediaQuery('(prefers-reduced-motion: reduce)');
};
