import { useState, useCallback, useRef, useEffect } from 'react';

/**
 * useProjectLoader - Manages project loading state with progress simulation
 *
 *
 * @returns loading state, progress, and control functions
 */
export const useProjectLoader = () => {
  const [loading, setLoading] = useState(false);
  const [progress, setProgress] = useState(0);
  const [projects, setProjects] = useState([]);
  const [completed, setCompleted] = useState(false);

  // Use ref to store interval ID to prevent stale closures
  const intervalRef = useRef<NodeJS.Timeout | null>(null);

  const loadProjects = useCallback(() => {
    // Clear any existing interval before starting new one
    if (intervalRef.current) {
      clearInterval(intervalRef.current);
      intervalRef.current = null;
    }

    setLoading(true);
    setProgress(0);
    setCompleted(false);

    intervalRef.current = setInterval(() => {
      setProgress((prev) => {
        if (prev >= 100) {
          // Clear interval when done
          if (intervalRef.current) {
            clearInterval(intervalRef.current);
            intervalRef.current = null;
          }
          setLoading(false);
          setCompleted(true);
          return 100;
        }
        return prev + Math.random() * 8 + 2;
      });
    }, 150);
  }, []);

  const reset = useCallback(() => {
    // Clear interval on reset
    if (intervalRef.current) {
      clearInterval(intervalRef.current);
      intervalRef.current = null;
    }

    setLoading(false);
    setProgress(0);
    setProjects([]);
    setCompleted(false);
  }, []);

  // Cleanup interval on component unmount
  useEffect(() => {
    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current);
        intervalRef.current = null;
      }
    };
  }, []);

  return {
    loading,
    progress,
    projects,
    completed,
    loadProjects,
    reset,
    setProgress
  };
};
