import { useEffect, useCallback, useRef } from 'react';
import { SCREEN_TYPES } from '../utils/constants';

interface UseSetupFlowProps {
  currentScreen: string;
  completed: boolean;
  saved: boolean;
  apiLoading: boolean;
  navigateTo: (screen: string) => void;
  completeCurrentStep: () => void;
  setCurrentStep: (updater: (prev: number) => number) => void;
  setLoading: (loading: boolean) => void;
}

interface SetupFlowTransition {
  from: string;
  to: string;
  delay: number;
  condition?: boolean;
  beforeNavigate?: () => void;
}

export const useSetupFlow = ({
  currentScreen,
  completed,
  saved,
  apiLoading,
  navigateTo,
  completeCurrentStep,
  setCurrentStep,
  setLoading
}: UseSetupFlowProps) => {
  // Track if we've started processing the network save
  const networkSaveAttemptedRef = useRef(false);
  const previousApiLoadingRef = useRef(apiLoading);

  // Detect when API loading transitions from true to false (save attempt completed)
  useEffect(() => {
    if (currentScreen === SCREEN_TYPES.NETWORK_SETUP) {
      // If API was loading and now it's not, a save attempt was completed
      if (previousApiLoadingRef.current && !apiLoading) {
        networkSaveAttemptedRef.current = true;
      }
      previousApiLoadingRef.current = apiLoading;
    }
  }, [apiLoading, currentScreen]);

  // Reset the flag when leaving network setup screen
  useEffect(() => {
    if (currentScreen !== SCREEN_TYPES.NETWORK_SETUP) {
      networkSaveAttemptedRef.current = false;
    }
  }, [currentScreen]);

  // Handle transitions when project loading completes
  useEffect(() => {
    if (!completed) return;

    const transitions: Record<string, SetupFlowTransition> = {
      [SCREEN_TYPES.LOADING_NETWORK]: {
        from: SCREEN_TYPES.LOADING_NETWORK,
        to: SCREEN_TYPES.NETWORK_SETUP,
        delay: 500
      },
      [SCREEN_TYPES.LOADING_AP]: {
        from: SCREEN_TYPES.LOADING_AP,
        to: SCREEN_TYPES.SUCCESS,
        delay: 2000
      },
      [SCREEN_TYPES.SUCCESS]: {
        from: SCREEN_TYPES.SUCCESS,
        to: SCREEN_TYPES.UPLOAD,
        delay: 4000,
        beforeNavigate: () => {
          completeCurrentStep(); // Complete current step first
          setCurrentStep(prev => prev + 1); // Then increment
        }
      }
    };

    const transition = transitions[currentScreen];
    if (!transition) return;

    const timer = setTimeout(() => {
      if (transition.beforeNavigate) {
        transition.beforeNavigate();
      }
      navigateTo(transition.to);
    }, transition.delay);

    return () => clearTimeout(timer);
  }, [completed, currentScreen, navigateTo, completeCurrentStep, setCurrentStep]);

  // Handle network save transitions
  useEffect(() => {
    if (currentScreen !== SCREEN_TYPES.NETWORK_SETUP) return;
    if (apiLoading) return; // Don't trigger while API is still loading

    // Only trigger transition if a save was actually attempted
    if (!networkSaveAttemptedRef.current) return;

    const timer = setTimeout(() => {
      if (saved) {
        setLoading(true);
        completeCurrentStep(); // Complete current step first
        setCurrentStep(prev => prev + 1); // Then increment
        navigateTo(SCREEN_TYPES.LOADING_AP);
        setLoading(false);
      }
    }, 2500);

    return () => clearTimeout(timer);
  }, [saved, apiLoading, currentScreen, setLoading, completeCurrentStep, setCurrentStep, navigateTo]);

  const handleSkipNetworkSetup = useCallback(() => {
    setLoading(true);
    const timer = setTimeout(() => {
      completeCurrentStep(); // Complete current step first
      setCurrentStep(prev => prev + 1); // Then increment
      navigateTo(SCREEN_TYPES.LOADING_AP);
      setLoading(false);
    }, 2500);

    return () => clearTimeout(timer);
  }, [setLoading, setCurrentStep, completeCurrentStep, navigateTo]);

  return {
    handleSkipNetworkSetup
  };
};
