import { useCallback } from 'react';
import { api } from '@/lib/api';
import type { Project } from '@/lib/api/types';
import { useApiData } from '@/shared/hooks';
import type { UseApiDataOptions } from '@/shared/hooks';

/**
 * Options for the useProjects hook
 */
interface UseProjectsOptions extends Omit<UseApiDataOptions<Project[]>, 'initialData'> {
  /** Auto-fetch projects on mount */
  autoFetch?: boolean;
  /** Refetch interval in milliseconds (0 = no polling) */
  refetchInterval?: number;
}

/**
 * useProjects Hook
 *
 * A convenient hook for managing projects with common patterns:
 * - Fetching projects list
 * - Creating new projects
 * - Updating projects
 * - Deleting projects
 * - Auto-refresh/polling
 * - Loading and error states
 *
 * @param options - Configuration options
 * @returns Project management utilities
 */
export const useProjects = (options: UseProjectsOptions = {}) => {
  const { autoFetch = true, refetchInterval = 0, ...restOptions } = options;

  // Fetch function that handles response extraction
  const fetchProjectsFn = useCallback(async (): Promise<Project[]> => {
    const response = await api.projects.list();

    // Backend might return data directly as array or wrapped in object
    const projectsArray = Array.isArray(response) ? response :
      Array.isArray(response.data) ? response.data : [];
    return projectsArray;
  }, []);

  // Use the generic useApiData hook
  const {
    data: projects,
    loading,
    error,
    refetch,
    addItem,
    removeItem,
    updateItem,
    clearError,
  } = useApiData(
    fetchProjectsFn,
    {
      autoFetch,
      refetchInterval,
      initialData: [],
      enableOptimisticUpdates: true,
      ...restOptions,
    }
  );

  /**
   * Fetch a single project by ID
   */
  const getProject = useCallback(async (id: string): Promise<Project | null> => {
    try {
      const response = await api.projects.get(id);
      return response.data || null;
    } catch (err) {
      console.error('Failed to fetch project:', err);
      return null;
    }
  }, []);

  /**
   * Create a new project
   */
  const createProject = useCallback(async (projectData: Partial<Project> | Record<string, unknown>): Promise<Project | null> => {
    try {
      const response = await api.projects.create(projectData);

      // Optimistically add to local state
      if (response.data) {
        addItem(response.data);
      }

      return response.data || null;
    } catch (err) {
      console.error('Failed to create project:', err);
      throw err;
    }
  }, [addItem]);

  /**
   * Update an existing project
   */
  const updateProject = useCallback(async (
    id: string,
    updates: Partial<Project>
  ): Promise<Project | null> => {
    try {
      const response = await api.projects.update(id, updates);

      // Optimistically update local state
      if (response.data) {
        updateItem(id, response.data);
      }

      return response.data || null;
    } catch (err) {
      console.error('Failed to update project:', err);
      throw err;
    }
  }, [updateItem]);

  /**
   * Delete a project
   */
  const deleteProject = useCallback(async (id: string): Promise<boolean> => {
    try {
      await api.projects.delete(id);

      // Optimistically remove from local state
      removeItem(id);

      return true;
    } catch (err) {
      console.error('Failed to delete project:', err);
      throw err;
    }
  }, [removeItem]);

  /**
   * Duplicate a project
   */
  const duplicateProject = useCallback(async (
    id: string,
    newName?: string
  ): Promise<Project | null> => {
    try {
      const response = await api.projects.duplicate(id, newName);

      // Optimistically add to local state
      if (response.data) {
        addItem(response.data);
      }

      return response.data || null;
    } catch (err) {
      console.error('Failed to duplicate project:', err);
      throw err;
    }
  }, [addItem]);

  return {
    // State
    projects,
    loading,
    error,

    // Actions
    fetchProjects: refetch,
    refetch,
    getProject,
    createProject,
    updateProject,
    deleteProject,
    duplicateProject,
    clearError,
  };
};
