/**
 * API Client
 * Centralized HTTP client with interceptors, retry logic, and error handling
 */

import type {
  APIClientConfig,
  RequestConfig,
  QueryParams,
  HTTPMethod,
} from './types';
import {
  APIError,
  TimeoutError,
  parseErrorFromResponse,
  handleFetchError,
} from './errors';
import { csrfManager, CSRF_CONFIG } from '../security/csrf';

/**
 * Build query string from params object
 */
function buildQueryString(params: QueryParams): string {
  const searchParams = new URLSearchParams();

  Object.entries(params).forEach(([key, value]) => {
    if (value !== null && value !== undefined) {
      searchParams.append(key, String(value));
    }
  });

  const queryString = searchParams.toString();
  return queryString ? `?${queryString}` : '';
}

/**
 * Sleep utility for retry delays
 */
function sleep(ms: number): Promise<void> {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

/**
 * Base API Client Class
 * Handles all HTTP communication with the backend
 */
export class APIClient {
  private baseURL: string;
  private defaultTimeout: number;
  private defaultHeaders: Record<string, string>;
  private requestInterceptors: NonNullable<APIClientConfig['interceptors']>['request'] = [];
  private responseInterceptors: NonNullable<APIClientConfig['interceptors']>['response'] = [];

  constructor(config: APIClientConfig) {
    this.baseURL = config.baseURL;
    this.defaultTimeout = config.timeout || 30000; // 30 seconds default
    this.defaultHeaders = {
      'Content-Type': 'application/json',
      ...config.headers,
    };

    if (config.interceptors) {
      this.requestInterceptors = config.interceptors.request || [];
      this.responseInterceptors = config.interceptors.response || [];
    }
  }

  /**
   * Add request interceptor
   */
  addRequestInterceptor(interceptor: NonNullable<APIClientConfig['interceptors']>['request'][0]) {
    this.requestInterceptors.push(interceptor);
  }

  /**
   * Add response interceptor
   */
  addResponseInterceptor(interceptor: NonNullable<APIClientConfig['interceptors']>['response'][0]) {
    this.responseInterceptors.push(interceptor);
  }

  /**
   * Build full URL from endpoint and query params
   */
  private buildURL(endpoint: string, params?: QueryParams): string {
    const url = endpoint.startsWith('http')
      ? endpoint
      : `${this.baseURL}${endpoint.startsWith('/') ? endpoint : `/${endpoint}`}`;

    return params ? `${url}${buildQueryString(params)}` : url;
  }

  /**
   * Apply request interceptors
   */
  private async applyRequestInterceptors(config: RequestConfig): Promise<RequestConfig> {
    let modifiedConfig = config;

    for (const interceptor of this.requestInterceptors) {
      modifiedConfig = await interceptor(modifiedConfig);
    }

    return modifiedConfig;
  }

  /**
   * Apply response interceptors
   */
  private async applyResponseInterceptors(response: Response): Promise<Response> {
    let modifiedResponse = response;

    for (const interceptor of this.responseInterceptors) {
      modifiedResponse = await interceptor(modifiedResponse);
    }

    return modifiedResponse;
  }

  /**
   * Execute HTTP request with timeout
   */
  private async executeRequest(
    url: string,
    config: RequestConfig
  ): Promise<Response> {
    const timeout = config.timeout || this.defaultTimeout;
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), timeout);

    try {
      const response = await fetch(url, {
        ...config,
        signal: controller.signal,
      });

      clearTimeout(timeoutId);
      return response;
    } catch (error) {
      clearTimeout(timeoutId);

      if (error instanceof Error && error.name === 'AbortError') {
        throw new TimeoutError(timeout);
      }

      throw handleFetchError(error);
    }
  }

  /**
   * Retry logic for failed requests
   */
  private async requestWithRetry(
    url: string,
    config: RequestConfig
  ): Promise<Response> {
    const retry = config.retry || { attempts: 0, delay: 1000 };
    let lastError: APIError | undefined;

    for (let attempt = 0; attempt <= retry.attempts; attempt++) {
      try {
        return await this.executeRequest(url, config);
      } catch (error) {
        lastError = error instanceof APIError ? error : handleFetchError(error);

        // Check if we should retry
        const shouldRetry =
          retry.shouldRetry?.(lastError) ?? lastError.shouldRetry();

        // Don't retry on last attempt or if shouldn't retry
        if (attempt === retry.attempts || !shouldRetry) {
          throw lastError;
        }

        // Wait before retrying with exponential backoff
        await sleep(retry.delay * Math.pow(2, attempt));
      }
    }

    throw lastError!;
  }

  /**
   * Core request method
   */
  async request<T = unknown>(
    method: HTTPMethod,
    endpoint: string,
    config: RequestConfig = {}
  ): Promise<T> {
    // Build URL
    const url = this.buildURL(endpoint, config.params as QueryParams);

    // Merge headers - but only include defaults that aren't explicitly undefined in config
    const headers = {
      ...this.defaultHeaders,
      ...config.headers,
    };

    // Remove any headers that are explicitly set to undefined (for FormData uploads)
    Object.keys(headers).forEach(key => {
      if (headers[key] === undefined) {
        delete headers[key];
      }
    });

    // Build request config
    let requestConfig: RequestConfig = {
      ...config,
      method,
      headers,
      credentials: 'include', // Always include cookies for HttpOnly auth
    };

    // Add CSRF protection for state-changing requests
    if (CSRF_CONFIG.PROTECTED_METHODS.includes(method as any)) {
      try {
        // Extract path from full URL for CSRF check
        const urlObj = new URL(url, typeof window !== 'undefined' ? window.location.origin : 'http://localhost');
        const path = urlObj.pathname;

        // Check if this endpoint needs CSRF protection
        if (csrfManager.requiresCSRF(method, path)) {
          const csrfToken = await csrfManager.getToken();
          requestConfig.headers = {
            ...requestConfig.headers,
            [CSRF_CONFIG.HEADER_NAME]: csrfToken,
          };
        }
      } catch (error) {
        console.error('Failed to add CSRF token:', error);
        // Don't throw - let the request proceed and let server reject if needed
      }
    }

    // Apply request interceptors
    requestConfig = await this.applyRequestInterceptors(requestConfig);

    // Execute request with retry logic
    let response = await this.requestWithRetry(url, requestConfig);

    // Apply response interceptors
    response = await this.applyResponseInterceptors(response);

    // Handle error responses
    if (!response.ok) {
      throw await parseErrorFromResponse(response);
    }

    // Parse response body
    const contentType = response.headers.get('content-type');
    if (contentType?.includes('application/json')) {
      return response.json();
    }

    // Return response as-is for non-JSON responses
    return response as any;
  }

  /**
   * GET request
   */
  async get<T = unknown>(endpoint: string, config?: RequestConfig): Promise<T> {
    return this.request<T>('GET', endpoint, config);
  }

  /**
   * POST request
   */
  async post<T = unknown>(
    endpoint: string,
    data?: any,
    config?: RequestConfig
  ): Promise<T> {
    return this.request<T>('POST', endpoint, {
      ...config,
      body: data ? JSON.stringify(data) : undefined,
    });
  }

  /**
   * PUT request
   */
  async put<T = unknown>(
    endpoint: string,
    data?: any,
    config?: RequestConfig
  ): Promise<T> {
    return this.request<T>('PUT', endpoint, {
      ...config,
      body: data ? JSON.stringify(data) : undefined,
    });
  }

  /**
   * PATCH request
   */
  async patch<T = unknown>(
    endpoint: string,
    data?: any,
    config?: RequestConfig
  ): Promise<T> {
    return this.request<T>('PATCH', endpoint, {
      ...config,
      body: data ? JSON.stringify(data) : undefined,
    });
  }

  /**
   * DELETE request
   */
  async delete<T = unknown>(
    endpoint: string,
    config?: RequestConfig
  ): Promise<T> {
    return this.request<T>('DELETE', endpoint, config);
  }

  /**
   * Upload file with multipart/form-data
   */
  async upload<T = unknown>(
    endpoint: string,
    file: File,
    additionalData?: Record<string, string>,
    config?: RequestConfig
  ): Promise<T> {
    const formData = new FormData();

    // Try different field names that backend might expect
    formData.append('pdf', file);  // Changed from 'file' to 'pdf'

    if (additionalData) {
      Object.entries(additionalData).forEach(([key, value]) => {
        formData.append(key, value);
      });
    }

    // Debug logging
    console.log('APIClient.upload - Details:', {
      endpoint,
      fileName: file.name,
      fileSize: file.size,
      fileType: file.type,
      additionalData,
      formDataEntries: Array.from(formData.entries()).map(([key, value]) => ({
        key,
        value: value instanceof File ? `File(${value.name}, ${value.size}bytes)` : value
      }))
    });

    // Remove Content-Type header to let browser set it with boundary
    // Set to undefined so it gets removed in the request() method
    const headers = {
      ...config?.headers,
      'Content-Type': undefined as any, // Explicitly undefined to override default
    };

    return this.request<T>('POST', endpoint, {
      ...config,
      headers,
      body: formData as any,
    });
  }
}

/**
 * Create default API client instance
 */
export function createAPIClient(config: Partial<APIClientConfig> = {}): APIClient {
  const baseURL = config.baseURL || process.env.NEXT_PUBLIC_API_BASE_URL || '/api';

  return new APIClient({
    baseURL,
    timeout: 30000,
    ...config,
  });
}