/**
 * Access Points API Endpoint Module
 * Handles all access point-related API operations
 */

import type { APIClient } from '../client';

/**
 * Access Point data structure
 */
export interface AccessPoint {
  id: string;
  name: string;
  mac?: string;
  ip?: string;
  ssid?: string;
  type?: string;
  status?: 'online' | 'offline' | 'unknown';
  network?: string;
  channel?: number;
  signal?: number;
  clients?: number;
  band?: string;
  location?: {
    x: number;
    y: number;
    floor?: string;
  };
  createdAt?: string;
  updatedAt?: string;
  [key: string]: any; // Allow additional properties
}

/**
 * Create access point data
 */
export interface CreateAccessPointData {
  name: string;
  mac?: string;
  ip?: string;
  ssid?: string;
  type?: string;
  network?: string;
  channel?: number;
  band?: string;
  location?: {
    x: number;
    y: number;
    floor?: string;
  };
  [key: string]: any;
}

/**
 * Update access point data
 */
export interface UpdateAccessPointData extends Partial<CreateAccessPointData> {
  id?: string;
  status?: 'online' | 'offline' | 'unknown';
  signal?: number;
  clients?: number;
}

/**
 * Access Points API class
 * Provides methods for access point CRUD operations
 */
export class AccessPointsAPI {
  constructor(private client: APIClient) {}

  /**
   * Get all access points
   * @returns Array of access points
   */
  async getAccessPoints(): Promise<AccessPoint[]> {
    return this.client.get<AccessPoint[]>('/edge/aps');
  }

  /**
   * Get a single access point by ID
   * @param id - Access Point ID
   * @returns Access Point object
   */
  async getAccessPoint(id: string): Promise<AccessPoint> {
    return this.client.get<AccessPoint>(`/edge/aps/${id}`);
  }

  /**
   * Create a new access point
   * @param data - Access Point data
   * @returns Created access point
   */
  async createAccessPoint(data: CreateAccessPointData): Promise<AccessPoint> {
    return this.client.post<AccessPoint>('/edge/aps', data);
  }

  /**
   * Update an existing access point
   * @param id - Access Point ID
   * @param data - Updated access point data
   * @returns Updated access point
   */
  async updateAccessPoint(id: string, data: UpdateAccessPointData): Promise<AccessPoint> {
    return this.client.put<AccessPoint>(`/edge/aps/${id}`, data);
  }

  /**
   * Delete an access point
   * @param id - Access Point ID
   * @returns Void
   */
  async deleteAccessPoint(id: string): Promise<void> {
    return this.client.delete<void>(`/edge/aps/${id}`);
  }

  /**
   * Patch an access point (partial update)
   * @param id - Access Point ID
   * @param data - Partial access point data
   * @returns Updated access point
   */
  async patchAccessPoint(id: string, data: Partial<UpdateAccessPointData>): Promise<AccessPoint> {
    return this.client.patch<AccessPoint>(`/edge/aps/${id}`, data);
  }

  /**
   * Get access points by network
   * @param networkId - Network ID
   * @returns Array of access points in the network
   */
  async getAccessPointsByNetwork(networkId: string): Promise<AccessPoint[]> {
    return this.client.get<AccessPoint[]>(`/edge/aps`, {
      params: { network: networkId }
    });
  }

  /**
   * Get access points by type
   * @param type - Access Point type
   * @returns Array of access points of the specified type
   */
  async getAccessPointsByType(type: string): Promise<AccessPoint[]> {
    return this.client.get<AccessPoint[]>(`/edge/aps`, {
      params: { type }
    });
  }

  /**
   * Get access point statistics
   * @param id - Access Point ID
   * @returns Access point statistics
   */
  async getAccessPointStats(id: string): Promise<{
    clients: number;
    bandwidth: { upload: number; download: number };
    uptime: number;
    [key: string]: any;
  }> {
    return this.client.get(`/edge/aps/${id}/stats`);
  }
}