/**
 * Devices API Endpoint Module
 * Handles all device-related API operations
 */

import type { APIClient } from '../client';

/**
 * Device data structure
 */
export interface Device {
  id: string;
  name: string;
  mac?: string;
  ip?: string;
  type?: string;
  status?: 'online' | 'offline' | 'unknown';
  network?: string;
  currentAP?: string;
  signal?: number;
  lastSeen?: string;
  createdAt?: string;
  updatedAt?: string;
  [key: string]: any; // Allow additional properties
}

/**
 * Create device data
 */
export interface CreateDeviceData {
  name: string;
  mac?: string;
  ip?: string;
  type?: string;
  network?: string;
  [key: string]: any;
}

/**
 * Update device data
 */
export interface UpdateDeviceData extends Partial<CreateDeviceData> {
  id?: string;
  status?: 'online' | 'offline' | 'unknown';
  currentAP?: string;
  signal?: number;
}

/**
 * Devices API class
 * Provides methods for device CRUD operations
 */
export class DevicesAPI {
  constructor(private client: APIClient) {}

  /**
   * Get all devices
   * @returns Array of devices
   */
  async getDevices(): Promise<Device[]> {
    return this.client.get<Device[]>('/edge/devices');
  }

  /**
   * Get a single device by ID
   * @param id - Device ID
   * @returns Device object
   */
  async getDevice(id: string): Promise<Device> {
    return this.client.get<Device>(`/edge/devices/${id}`);
  }

  /**
   * Create a new device
   * @param data - Device data
   * @returns Created device
   */
  async createDevice(data: CreateDeviceData): Promise<Device> {
    return this.client.post<Device>('/edge/devices', data);
  }

  /**
   * Update an existing device
   * @param id - Device ID
   * @param data - Updated device data
   * @returns Updated device
   */
  async updateDevice(id: string, data: UpdateDeviceData): Promise<Device> {
    return this.client.put<Device>(`/edge/devices/${id}`, data);
  }

  /**
   * Delete a device
   * @param id - Device ID
   * @returns Void
   */
  async deleteDevice(id: string): Promise<void> {
    return this.client.delete<void>(`/edge/devices/${id}`);
  }

  /**
   * Patch a device (partial update)
   * @param id - Device ID
   * @param data - Partial device data
   * @returns Updated device
   */
  async patchDevice(id: string, data: Partial<UpdateDeviceData>): Promise<Device> {
    return this.client.patch<Device>(`/edge/devices/${id}`, data);
  }

  /**
   * Get devices by network
   * @param networkId - Network ID
   * @returns Array of devices in the network
   */
  async getDevicesByNetwork(networkId: string): Promise<Device[]> {
    return this.client.get<Device[]>(`/edge/devices`, {
      params: { network: networkId }
    });
  }

  /**
   * Get devices by access point
   * @param apId - Access Point ID
   * @returns Array of devices connected to the AP
   */
  async getDevicesByAccessPoint(apId: string): Promise<Device[]> {
    return this.client.get<Device[]>(`/edge/devices`, {
      params: { currentAP: apId }
    });
  }
}