/**
 * Networks API Endpoint Module
 * Handles all network-related API operations
 */

import type { APIClient } from '../client';

/**
 * Network data structure
 */
export interface Network {
  id: string;
  name: string;
  ssid?: string;
  password?: string;
  security?: string;
  band?: string;
  channel?: number;
  enabled?: boolean;
  createdAt?: string;
  updatedAt?: string;
  [key: string]: any; // Allow additional properties
}

/**
 * Create network data
 */
export interface CreateNetworkData {
  name: string;
  ssid?: string;
  password?: string;
  security?: string;
  band?: string;
  channel?: number;
  enabled?: boolean;
  [key: string]: any;
}

/**
 * Update network data
 */
export interface UpdateNetworkData extends Partial<CreateNetworkData> {
  id?: string;
}

/**
 * Networks API class
 * Provides methods for network CRUD operations
 */
export class NetworksAPI {
  constructor(private client: APIClient) {}

  /**
   * Get all networks
   * @returns Array of networks
   */
  async getNetworks(): Promise<Network[]> {
    return this.client.get<Network[]>('/edge/networks');
  }

  /**
   * Get a single network by ID
   * @param id - Network ID
   * @returns Network object
   */
  async getNetwork(id: string): Promise<Network> {
    return this.client.get<Network>(`/edge/networks/${id}`);
  }

  /**
   * Create a new network
   * @param data - Network data
   * @returns Created network
   */
  async createNetwork(data: CreateNetworkData): Promise<Network> {
    return this.client.post<Network>('/edge/networks', data);
  }

  /**
   * Update an existing network
   * @param id - Network ID
   * @param data - Updated network data
   * @returns Updated network
   */
  async updateNetwork(id: string, data: UpdateNetworkData): Promise<Network> {
    return this.client.put<Network>(`/edge/networks/${id}`, data);
  }

  /**
   * Delete a network
   * @param id - Network ID
   * @returns Void
   */
  async deleteNetwork(id: string): Promise<void> {
    return this.client.delete<void>(`/edge/networks/${id}`);
  }

  /**
   * Patch a network (partial update)
   * @param id - Network ID
   * @param data - Partial network data
   * @returns Updated network
   */
  async patchNetwork(id: string, data: Partial<UpdateNetworkData>): Promise<Network> {
    return this.client.patch<Network>(`/edge/networks/${id}`, data);
  }
}