/**
 * Projects API Endpoints
 * All project-related API calls
 */

import type { APIClient } from '../client';
import type {
  Project,
  CreateProjectRequest,
  UpdateProjectRequest,
  APIResponse,
  PaginatedResponse,
  QueryParams,
} from '../types';

export interface ProjectsQueryParams extends QueryParams {
  page?: number;
  pageSize?: number;
  search?: string;
  sortBy?: 'name' | 'timestamp' | 'updatedAt';
  sortOrder?: 'asc' | 'desc';
}

export class ProjectsAPI {
  constructor(private client: APIClient) {}

  /**
   * Get all projects
   */
  async list(params?: ProjectsQueryParams): Promise<APIResponse<Project[]>> {
    return this.client.get<APIResponse<Project[]>>('/projects', {
      params: params as any,
    });
  }

  /**
   * Get paginated projects
   */
  async listPaginated(
    params?: ProjectsQueryParams
  ): Promise<PaginatedResponse<Project>> {
    return this.client.get<PaginatedResponse<Project>>('/projects/paginated', {
      params: params as any,
    });
  }

  /**
   * Get project by ID
   */
  async get(id: string): Promise<APIResponse<Project>> {
    return this.client.get<APIResponse<Project>>(`/projects/${id}`);
  }

  /**
   * Get project by name
   */
  async getByName(name: string): Promise<APIResponse<Project>> {
    return this.client.get<APIResponse<Project>>(`/projects/by-name/${encodeURIComponent(name)}`);
  }

  /**
   * Create new project
   */
  async create(data: CreateProjectRequest): Promise<APIResponse<Project>> {
    return this.client.post<APIResponse<Project>>('/projects', data);
  }

  /**
   * Update existing project
   */
  async update(id: string, data: Partial<UpdateProjectRequest>): Promise<APIResponse<Project>> {
    return this.client.put<APIResponse<Project>>(`/projects/${id}`, data);
  }

  /**
   * Partially update project
   */
  async patch(id: string, data: Partial<UpdateProjectRequest>): Promise<APIResponse<Project>> {
    return this.client.patch<APIResponse<Project>>(`/projects/${id}`, data);
  }

  /**
   * Delete project
   */
  async delete(id: string): Promise<APIResponse<void>> {
    return this.client.delete<APIResponse<void>>(`/projects/${id}`);
  }

  /**
   * Delete project by name
   */
  async deleteByName(name: string): Promise<APIResponse<void>> {
    return this.client.delete<APIResponse<void>>(`/projects/by-name/${encodeURIComponent(name)}`);
  }

  /**
   * Duplicate project
   */
  async duplicate(id: string, newName?: string): Promise<APIResponse<Project>> {
    return this.client.post<APIResponse<Project>>(`/projects/${id}/duplicate`, {
      name: newName,
    });
  }

  /**
   * Export project as JSON
   */
  async export(id: string): Promise<Blob> {
    const response = await this.client.request<Response>('GET', `/projects/${id}/export`, {
      headers: {
        Accept: 'application/json',
      },
    });

    return response.blob();
  }

  /**
   * Import project from JSON
   */
  async import(file: File): Promise<APIResponse<Project>> {
    return this.client.upload<APIResponse<Project>>('/projects/import', file);
  }

  /**
   * Get project statistics
   */
  async getStatistics(id: string): Promise<
    APIResponse<{
      accessPointsCount: number;
      pagesCount: number;
      lastModified: number;
      fileSize: number;
    }>
  > {
    return this.client.get<any>(`/projects/${id}/statistics`);
  }

  /**
   * Batch delete projects
   */
  async batchDelete(ids: string[]): Promise<APIResponse<{ deleted: number }>> {
    return this.client.post<APIResponse<{ deleted: number }>>('/projects/batch-delete', {
      ids,
    });
  }

  /**
   * Search projects
   */
  async search(query: string, limit?: number): Promise<APIResponse<Project[]>> {
    return this.client.get<APIResponse<Project[]>>('/projects/search', {
      params: { q: query, limit } as any,
    });
  }
}