/**
 * File Upload API Endpoints
 * All file upload-related API calls
 */

import type { APIClient } from '../client';
import type { UploadResponse, APIResponse } from '../types';

export interface UploadProgress {
  loaded: number;
  total: number;
  percentage: number;
}

export type UploadProgressCallback = (progress: UploadProgress) => void;

export class UploadsAPI {
  constructor(private client: APIClient) {}

  /**
   * Upload PDF file
   */
  async uploadPDF(
    file: File,
    onProgress?: UploadProgressCallback
  ): Promise<APIResponse<UploadResponse>> {
    // TODO: Implement progress tracking with XMLHttpRequest
    // Try without additional metadata first - backend might not expect it
    return this.client.upload<APIResponse<UploadResponse>>(
      '/projects/uploads/pdf',
      file
    );
  }

  /**
   * Upload image file
   */
  async uploadImage(
    file: File,
    onProgress?: UploadProgressCallback
  ): Promise<APIResponse<UploadResponse>> {
    return this.client.upload<APIResponse<UploadResponse>>(
      '/uploads/image',
      file,
      {
        type: 'image',
        originalName: file.name,
      }
    );
  }

  /**
   * Upload generic file
   */
  async uploadFile(
    file: File,
    endpoint: string = '/uploads/file',
    additionalData?: Record<string, string>,
    onProgress?: UploadProgressCallback
  ): Promise<APIResponse<UploadResponse>> {
    return this.client.upload<APIResponse<UploadResponse>>(
      endpoint,
      file,
      additionalData
    );
  }

  /**
   * Delete uploaded file
   */
  async deleteFile(path: string): Promise<APIResponse<void>> {
    return this.client.delete<APIResponse<void>>('/uploads/delete', {
      params: { path } as any,
    });
  }

  /**
   * Get file metadata
   */
  async getFileMetadata(path: string): Promise<
    APIResponse<{
      name: string;
      size: number;
      mimeType: string;
      uploadedAt: number;
    }>
  > {
    return this.client.get<any>('/uploads/metadata', {
      params: { path } as any,
    });
  }

  /**
   * Get file URL
   */
  getFileURL(path: string): string {
    const baseURL = this.client['baseURL'];
    return `${baseURL}/${path}`;
  }

  /**
   * Download file
   */
  async downloadFile(path: string, filename?: string): Promise<void> {
    const response = await this.client.request<Response>('GET', `/uploads/download`, {
      params: { path } as any,
    });

    const blob = await response.blob();
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename || path.split('/').pop() || 'download';
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
  }

  /**
   * Validate file before upload
   */
  validateFile(
    file: File,
    options: {
      maxSize?: number; // in bytes
      allowedTypes?: string[]; // MIME types
      allowedExtensions?: string[];
    } = {}
  ): { valid: boolean; error?: string } {
    const { maxSize, allowedTypes, allowedExtensions } = options;

    // Check file size
    if (maxSize && file.size > maxSize) {
      const maxSizeMB = (maxSize / (1024 * 1024)).toFixed(2);
      return {
        valid: false,
        error: `File size exceeds ${maxSizeMB}MB limit`,
      };
    }

    // Check MIME type
    if (allowedTypes && !allowedTypes.includes(file.type)) {
      return {
        valid: false,
        error: `File type ${file.type} is not allowed`,
      };
    }

    // Check file extension
    if (allowedExtensions) {
      const extension = file.name.split('.').pop()?.toLowerCase();
      if (!extension || !allowedExtensions.includes(extension)) {
        return {
          valid: false,
          error: `File extension .${extension} is not allowed`,
        };
      }
    }

    return { valid: true };
  }

  /**
   * Upload PDF with validation
   */
  async uploadPDFWithValidation(
    file: File,
    maxSizeMB: number = 50,
    onProgress?: UploadProgressCallback
  ): Promise<APIResponse<UploadResponse>> {
    // Validate file
    const validation = this.validateFile(file, {
      maxSize: maxSizeMB * 1024 * 1024,
      allowedTypes: ['application/pdf'],
      allowedExtensions: ['pdf'],
    });

    if (!validation.valid) {
      throw new Error(validation.error);
    }

    // Upload file
    return this.uploadPDF(file, onProgress);
  }
}
