/**
 * API Error Handling
 * Centralized error types and handling for API requests
 */

import type { APIErrorType } from './types';

/**
 * Custom API Error class with enhanced error information
 */
export class APIError extends Error implements APIErrorType {
  public readonly status?: number;
  public readonly code?: string;
  public readonly details?: Record<string, any>;
  public readonly timestamp: number;

  constructor(
    message: string,
    status?: number,
    code?: string,
    details?: Record<string, any>
  ) {
    super(message);
    this.name = 'APIError';
    this.status = status;
    this.code = code;
    this.details = details;
    this.timestamp = Date.now();

    // Maintains proper stack trace for where error was thrown (V8 only)
    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, APIError);
    }
  }

  /**
   * Convert error to JSON format
   */
  toJSON(): APIErrorType {
    return {
      message: this.message,
      code: this.code,
      status: this.status,
      details: this.details,
    };
  }

  /**
   * Check if error is a network error
   */
  isNetworkError(): boolean {
    return !this.status || this.status === 0;
  }

  /**
   * Check if error is a client error (4xx)
   */
  isClientError(): boolean {
    return !!this.status && this.status >= 400 && this.status < 500;
  }

  /**
   * Check if error is a server error (5xx)
   */
  isServerError(): boolean {
    return !!this.status && this.status >= 500;
  }

  /**
   * Check if error is authentication related
   */
  isAuthError(): boolean {
    return this.status === 401 || this.status === 403;
  }

  /**
   * Check if request should be retried
   */
  shouldRetry(): boolean {
    // Retry on network errors and 5xx server errors
    return this.isNetworkError() || this.isServerError();
  }
}

/**
 * Timeout Error
 */
export class TimeoutError extends APIError {
  constructor(timeout: number) {
    super(`Request timeout after ${timeout}ms`, 408, 'TIMEOUT');
    this.name = 'TimeoutError';
  }
}

/**
 * Network Error
 */
export class NetworkError extends APIError {
  constructor(message: string = 'Network request failed') {
    super(message, 0, 'NETWORK_ERROR');
    this.name = 'NetworkError';
  }
}

/**
 * Validation Error
 */
export class ValidationError extends APIError {
  constructor(message: string, details?: Record<string, any>) {
    super(message, 400, 'VALIDATION_ERROR', details);
    this.name = 'ValidationError';
  }
}

/**
 * Authentication Error
 */
export class AuthenticationError extends APIError {
  constructor(message: string = 'Authentication required') {
    super(message, 401, 'AUTHENTICATION_ERROR');
    this.name = 'AuthenticationError';
  }
}

/**
 * Authorization Error
 */
export class AuthorizationError extends APIError {
  constructor(message: string = 'Insufficient permissions') {
    super(message, 403, 'AUTHORIZATION_ERROR');
    this.name = 'AuthorizationError';
  }
}

/**
 * Not Found Error
 */
export class NotFoundError extends APIError {
  constructor(resource: string = 'Resource') {
    super(`${resource} not found`, 404, 'NOT_FOUND');
    this.name = 'NotFoundError';
  }
}

/**
 * Rate Limit Error
 */
export class RateLimitError extends APIError {
  constructor(retryAfter?: number) {
    super(
      'Too many requests. Please try again later.',
      429,
      'RATE_LIMIT',
      retryAfter ? { retryAfter } : undefined
    );
    this.name = 'RateLimitError';
  }
}

/**
 * Parse error from Response object
 */
export async function parseErrorFromResponse(response: Response): Promise<APIError> {
  const status = response.status;
  const statusText = response.statusText;

  try {
    const data = await response.json();
    const message = data.error || data.message || statusText || 'Request failed';
    const code = data.code || `HTTP_${status}`;
    const details = data.details || {};

    // Return specific error types based on status code
    switch (status) {
    case 400:
      return new ValidationError(message, details);
    case 401:
      return new AuthenticationError(message);
    case 403:
      return new AuthorizationError(message);
    case 404:
      return new NotFoundError(message);
    case 429:
      return new RateLimitError(details.retryAfter);
    default:
      return new APIError(message, status, code, details);
    }
  } catch {
    // If the response body is not JSON, create an error from status
    return new APIError(
      statusText || `Request failed with status ${status}`,
      status,
      `HTTP_${status}`
    );
  }
}

/**
 * Handle fetch errors (network errors, timeouts, etc.)
 */
export function handleFetchError(error: unknown): APIError {
  if (error instanceof APIError) {
    return error;
  }

  if (error instanceof Error) {
    // Check for common network error messages
    if (
      error.message.includes('fetch') ||
      error.message.includes('network') ||
      error.message.includes('Failed to fetch')
    ) {
      return new NetworkError(error.message);
    }

    return new APIError(error.message, undefined, 'UNKNOWN_ERROR');
  }

  return new APIError('An unknown error occurred', undefined, 'UNKNOWN_ERROR');
}
