/**
 * API Client - Main Entry Point
 * Centralized API client with all endpoints
 */

import { APIClient, createAPIClient } from './client';
import { AuthAPI } from './endpoints/auth';
import { ProjectsAPI } from './endpoints/projects';
import { UploadsAPI } from './endpoints/uploads';
import { NetworksAPI } from './endpoints/networks';
import { DevicesAPI } from './endpoints/devices';
import { AccessPointsAPI } from './endpoints/accessPoints';
import Cookies from 'js-cookie';

/**
 * Main API class that combines all endpoint modules
 */
export class API {
  private client: APIClient;

  public auth: AuthAPI;
  public projects: ProjectsAPI;
  public uploads: UploadsAPI;
  public networks: NetworksAPI;
  public devices: DevicesAPI;
  public accessPoints: AccessPointsAPI;

  constructor(client: APIClient) {
    this.client = client;

    // Initialize endpoint modules
    this.auth = new AuthAPI(client);
    this.projects = new ProjectsAPI(client);
    this.uploads = new UploadsAPI(client);
    this.networks = new NetworksAPI(client);
    this.devices = new DevicesAPI(client);
    this.accessPoints = new AccessPointsAPI(client);
  }

  /**
   * Get underlying client instance (for advanced usage)
   */
  getClient(): APIClient {
    return this.client;
  }
}

/**
 * Create default API instance with auth interceptor
 */
function createDefaultAPI(): API {
  const client = createAPIClient({
    baseURL: process.env.NEXT_PUBLIC_API_BASE_URL || '/api',
    timeout: 30000,
  });

  // Add auth token interceptor
  client.addRequestInterceptor(async (config: { headers: any; }) => {
    // Get token from cookie (legacy system)
    const token = Cookies.get('apiKey');

    if (token) {
      config.headers = {
        ...config.headers,
        Authorization: `Bearer ${token}`,
      };
    }

    return config;
  });

  // Add response error handler for auth errors
  client.addResponseInterceptor(async (response: { status: number; }) => {
    // Handle 401 Unauthorized - clear token and redirect to login
    if (response.status === 401) {
      Cookies.remove('apiKey');
      Cookies.remove('userEmail');

      // Only redirect if we're not already on login page
      if (typeof window !== 'undefined' && !window.location.pathname.includes('/login')) {
        window.location.href = '/login';
      }
    }

    return response;
  });

  return new API(client);
}

/**
 * Default API instance
 * Use this for most API calls
 */
export const api = createDefaultAPI();
/**
 * Create custom API instance
 * Use this when you need custom configuration
 */
export function createAPI(baseURL?: string): API {
  const client = createAPIClient({ baseURL });
  return new API(client);
}

// Export all types
export * from './types';
export * from './errors';
export * from './client';

// Export endpoint classes for custom usage
export { AuthAPI } from './endpoints/auth';
export { ProjectsAPI } from './endpoints/projects';
export { UploadsAPI } from './endpoints/uploads';
export { NetworksAPI } from './endpoints/networks';
export { DevicesAPI } from './endpoints/devices';
export { AccessPointsAPI } from './endpoints/accessPoints';
