/**
 * API Client Type Definitions
 * Centralized type definitions for all API interactions
 */

// ==================== Common Types ====================

export interface APIResponse<T = unknown> {
  success: boolean;
  data?: T;
  error?: string;
  message?: string;
}

export interface PaginatedResponse<T> {
  data: T[];
  pagination: {
    page: number;
    pageSize: number;
    total: number;
    totalPages: number;
  };
}

export interface APIErrorType {
  message: string;
  code?: string;
  status?: number;
  details?: Record<string, any>;
}

// ==================== Auth Types ====================

export interface LoginRequest {
  username: string;
  password: string;
  rememberMe?: boolean;
}

export interface LoginResponse {
  success: boolean;
  apiKey?: string;
  user?: {
    id: string;
      username: string;
    name?: string;
  };
  requiresMFA?: boolean;
}

export interface User {
  id: string;
  username: string;
  name?: string;
  role?: string;
}

// ==================== Project Types ====================

export interface AccessPoint {
  id: string;
  type: string;
  name: string;
  floor: string;
  x: number;
  y: number;
  relativeX?: number;
  relativeY?: number;
  page: number;
  deviceDetails?: any;
}

export interface PDFFile {
  name: string;
  size: number;
  originalName: string;
  uploadedPath: string | null;
  file?: File;
}

export interface ViewportState {
  zoomLevel: number;
  panOffset: { x: number; y: number };
  rotation?: number;
}

export interface Project {
  id?: string;
  name: string;
  accessPoints: AccessPoint[];
  currentPage: number;
  timestamp: number;
  pageRotation?: number;
  pdfFile: PDFFile;
  viewportState?: {
    zoomLevel: number;
    panOffset: { x: number; y: number };
    pageViewports?: Record<number, ViewportState>;
  };
}

export interface CreateProjectRequest {
  name: string;
  accessPoints: AccessPoint[];
  currentPage: number;
  timestamp: number;
  pageRotation?: number;
  pdfFile: PDFFile;
  viewportState?: any;
}

export interface UpdateProjectRequest extends Partial<CreateProjectRequest> {
  id: string;
}

// ==================== Upload Types ====================

export interface UploadResponse {
  success: boolean;
  path: string;
  url?: string;
  filename: string;
}

// ==================== Request Configuration ====================

export interface RequestConfig extends RequestInit {
  timeout?: number;
  retry?: {
    attempts: number;
    delay: number;
    shouldRetry?: (error: APIErrorType) => boolean;
  };
}

export interface APIClientConfig {
  baseURL: string;
  timeout?: number;
  headers?: Record<string, string>;
  interceptors?: {
    request?: RequestInterceptor[];
    response?: ResponseInterceptor[];
  };
}

export type RequestInterceptor = (
  config: RequestConfig
) => RequestConfig | Promise<RequestConfig>;

export type ResponseInterceptor = (
  response: Response
) => Response | Promise<Response>;

// ==================== HTTP Methods ====================

export type HTTPMethod = 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE';

// ==================== Query Parameters ====================

export type QueryParams = Record<string, string | number | boolean | null | undefined>;
