/**
 * Unified Error Handler
 * Centralized error handling, logging, and user notification
 */

export enum ErrorSeverity {
  INFO = 'info',
  WARNING = 'warning',
  ERROR = 'error',
  CRITICAL = 'critical'
}

export enum ErrorCategory {
  PDF_PROCESSING = 'pdf_processing',
  NETWORK = 'network',
  API = 'api',
  VALIDATION = 'validation',
  FILE_UPLOAD = 'file_upload',
  PROJECT_MANAGEMENT = 'project_management',
  UNKNOWN = 'unknown'
}

export interface ErrorContext {
  category: ErrorCategory;
  severity: ErrorSeverity;
  userMessage?: string;
  technicalDetails?: any;
  retry?: boolean;
  timestamp?: number;
}

export interface ErrorHandlerOptions {
  showToast?: boolean;
  logToConsole?: boolean;
  setLoadingError?: (message: string | null) => void;
  addToast?: (toast: { title: string; color: string; text: string }) => void;
}

export class AppError extends Error {
  category: ErrorCategory;
  severity: ErrorSeverity;
  userMessage: string;
  technicalDetails: any;
  retry: boolean;
  timestamp: number;

  constructor(
    message: string,
    context: Partial<ErrorContext> = {}
  ) {
    super(message);
    this.name = 'AppError';
    this.category = context.category || ErrorCategory.UNKNOWN;
    this.severity = context.severity || ErrorSeverity.ERROR;
    this.userMessage = context.userMessage || message;
    this.technicalDetails = context.technicalDetails;
    this.retry = context.retry || false;
    this.timestamp = context.timestamp || Date.now();
  }
}

/**
 * Maps error categories to user-friendly messages
 */
const getCategoryMessage = (category: ErrorCategory): string => {
  const messages: Record<ErrorCategory, string> = {
    [ErrorCategory.PDF_PROCESSING]: 'Failed to process PDF file',
    [ErrorCategory.NETWORK]: 'Network connection error',
    [ErrorCategory.API]: 'Server communication error',
    [ErrorCategory.VALIDATION]: 'Invalid input',
    [ErrorCategory.FILE_UPLOAD]: 'Failed to upload file',
    [ErrorCategory.PROJECT_MANAGEMENT]: 'Project operation failed',
    [ErrorCategory.UNKNOWN]: 'An unexpected error occurred'
  };

  return messages[category];
};

/**
 * Maps severity to toast color
 */
const getSeverityColor = (severity: ErrorSeverity): string => {
  const colors: Record<ErrorSeverity, string> = {
    [ErrorSeverity.INFO]: 'primary',
    [ErrorSeverity.WARNING]: 'warning',
    [ErrorSeverity.ERROR]: 'danger',
    [ErrorSeverity.CRITICAL]: 'danger'
  };

  return colors[severity];
};

/**
 * Unified error handler function
 */
export const handleError = (
  error: Error | AppError | unknown,
  options: ErrorHandlerOptions = {}
): AppError => {
  const {
    showToast = true,
    logToConsole = true,
    setLoadingError,
    addToast
  } = options;

  let appError: AppError;

  // Convert to AppError if not already
  if (error instanceof AppError) {
    appError = error;
  } else if (error instanceof Error) {
    appError = new AppError(error.message, {
      category: categorizeError(error),
      technicalDetails: error
    });
  } else {
    appError = new AppError(String(error), {
      category: ErrorCategory.UNKNOWN,
      technicalDetails: error
    });
  }

  // Log to console if enabled
  if (logToConsole) {
    const logLevel = appError.severity === ErrorSeverity.CRITICAL || appError.severity === ErrorSeverity.ERROR
      ? 'error'
      : 'warn';

    console[logLevel](`[${appError.category}] ${appError.message}`, {
      severity: appError.severity,
      timestamp: new Date(appError.timestamp).toISOString(),
      technicalDetails: appError.technicalDetails
    });
  }

  // Set loading error if handler provided
  if (setLoadingError) {
    setLoadingError(appError.userMessage);
  }

  // Show toast notification if handler provided and enabled
  if (showToast && addToast) {
    addToast({
      title: getCategoryMessage(appError.category),
      color: getSeverityColor(appError.severity),
      text: appError.userMessage
    });
  }

  return appError;
};

/**
 * Attempts to categorize an error based on its properties
 */
const categorizeError = (error: Error): ErrorCategory => {
  const message = error.message.toLowerCase();

  if (message.includes('pdf') || message.includes('pdfjslib')) {
    return ErrorCategory.PDF_PROCESSING;
  }
  if (message.includes('fetch') || message.includes('network') || message.includes('connection')) {
    return ErrorCategory.NETWORK;
  }
  if (message.includes('api') || message.includes('status')) {
    return ErrorCategory.API;
  }
  if (message.includes('upload')) {
    return ErrorCategory.FILE_UPLOAD;
  }
  if (message.includes('validation') || message.includes('invalid')) {
    return ErrorCategory.VALIDATION;
  }
  if (message.includes('project') || message.includes('save') || message.includes('load')) {
    return ErrorCategory.PROJECT_MANAGEMENT;
  }

  return ErrorCategory.UNKNOWN;
};

/**
 * Helper to create PDF processing errors
 */
export const createPdfError = (message: string, details?: any): AppError => {
  return new AppError(message, {
    category: ErrorCategory.PDF_PROCESSING,
    severity: ErrorSeverity.ERROR,
    userMessage: 'Failed to process PDF file. Please ensure it\'s a valid PDF.',
    technicalDetails: details
  });
};

/**
 * Helper to create network errors
 */
export const createNetworkError = (message: string, details?: any): AppError => {
  return new AppError(message, {
    category: ErrorCategory.NETWORK,
    severity: ErrorSeverity.WARNING,
    userMessage: 'Network connection error. Please check your connection and try again.',
    technicalDetails: details,
    retry: true
  });
};

/**
 * Helper to create API errors
 */
export const createApiError = (
  message: string,
  statusCode?: number,
  details?: any
): AppError => {
  const userMessage = statusCode === 404
    ? 'Resource not found'
    : statusCode === 403
      ? 'Access denied'
      : statusCode === 500
        ? 'Server error. Please try again later.'
        : 'Failed to communicate with server';

  return new AppError(message, {
    category: ErrorCategory.API,
    severity: statusCode && statusCode >= 500 ? ErrorSeverity.ERROR : ErrorSeverity.WARNING,
    userMessage,
    technicalDetails: { statusCode, ...details },
    retry: statusCode && statusCode >= 500
  });
};

/**
 * Helper to create validation errors
 */
export const createValidationError = (message: string, field?: string): AppError => {
  return new AppError(message, {
    category: ErrorCategory.VALIDATION,
    severity: ErrorSeverity.INFO,
    userMessage: message,
    technicalDetails: { field }
  });
};

/**
 * Wraps an async function with error handling
 */
export const withErrorHandler = <T extends (...args: any[]) => Promise<any>>(
  fn: T,
  options: ErrorHandlerOptions = {}
): T => {
  return (async (...args: Parameters<T>): Promise<ReturnType<T>> => {
    try {
      return await fn(...args);
    } catch (error) {
      handleError(error, options);
      throw error;
    }
  }) as T;
};