/**
 * File validation utilities for upload restrictions
 */

export interface FileValidationOptions {
  maxSize?: number; // in bytes
  allowedTypes?: string[];
  allowedExtensions?: string[];
}

export interface FileValidationResult {
  valid: boolean;
  error?: string;
  file?: File;
}

import { config } from './config';

// Default file size limits (in MB)
export const FILE_SIZE_LIMITS = {
  PDF: (config.features.maxPdfSizeMB || 50) * 1024 * 1024,
  IMAGE: 10 * 1024 * 1024, // 10MB
  DOCUMENT: 25 * 1024 * 1024, // 25MB
  DEFAULT: 10 * 1024 * 1024 // 10MB
};

// Allowed MIME types
export const ALLOWED_MIME_TYPES = {
  PDF: ['application/pdf'],
  IMAGE: ['image/jpeg', 'image/jpg', 'image/png'],
  DOCUMENT: [
    'application/pdf',
  ]
};

/**
 * Validates a file based on size, type, and extension restrictions
 */
export const validateFile = (
  file: File,
  options: FileValidationOptions = {}
): FileValidationResult => {
  const {
    maxSize = FILE_SIZE_LIMITS.DEFAULT,
    allowedTypes = [],
    allowedExtensions = []
  } = options;

  // Check if file exists
  if (!file) {
    return {
      valid: false,
      error: 'No file provided'
    };
  }

  // Validate file size
  if (file.size > maxSize) {
    const maxSizeMB = (maxSize / (1024 * 1024)).toFixed(2);
    const fileSizeMB = (file.size / (1024 * 1024)).toFixed(2);
    return {
      valid: false,
      error: `File size (${fileSizeMB}MB) exceeds maximum allowed size of ${maxSizeMB}MB`
    };
  }

  // Validate file type (MIME type)
  if (allowedTypes.length > 0 && !allowedTypes.includes(file.type)) {
    return {
      valid: false,
      error: `File type "${file.type}" is not allowed. Allowed types: ${allowedTypes.join(', ')}`
    };
  }

  // Validate file extension
  if (allowedExtensions.length > 0) {
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    if (!fileExtension || !allowedExtensions.includes(fileExtension)) {
      return {
        valid: false,
        error: `File extension ".${fileExtension}" is not allowed. Allowed extensions: ${allowedExtensions.join(', ')}`
      };
    }
  }

  // Additional validation: Check for empty files
  if (file.size === 0) {
    return {
      valid: false,
      error: 'File is empty'
    };
  }

  return {
    valid: true,
    file
  };
};

/**
 * Validates PDF files specifically
 */
export const validatePdfFile = (file: File): FileValidationResult => {
  return validateFile(file, {
    maxSize: FILE_SIZE_LIMITS.PDF,
    allowedTypes: ALLOWED_MIME_TYPES.PDF,
    allowedExtensions: ['pdf']
  });
};
