import { api } from "@/lib/api";

export interface AuthResult {
  success: boolean;
  token?: string;
  error?: string;
  requiresMFA?: boolean;
}

/**
 * Check if user has valid session by calling backend
 * Backend validates HttpOnly cookies
 */
export const hasValidSession = async (): Promise<boolean> => {
  try {
    const response = await api.auth.verifySession();
    return response.data?.valid ?? false;
  } catch (error) {
    console.error('[AUTH] Session verification failed:', error);
    return false;
  }
};

/**
 * Login user with email and password
 * Backend sets HttpOnly cookies on success
 */
export const login = async (
  email: string,
  password: string,
  rememberMe: boolean = false
): Promise<AuthResult> => {
  try {
    const response = await api.auth.login({ email, password, rememberMe });

    return {
      success: response.success,
      token: response.token,
      error: response.error,
      requiresMFA: response.requiresMFA,
    };
  } catch (error) {
    console.error('[AUTH] Login error:', error);
    return {
      success: false,
      error: 'Network error. Please try again.',
    };
  }
};

/**
 * Logout user
 * Backend clears HttpOnly cookies
 */
export const logout = async (): Promise<boolean> => {
  try {
    const response = await api.auth.logout();
    return response.success;
  } catch (error) {
    console.error('[AUTH] Logout error:', error);
    return false;
  }
};

/**
 * Security Event Types (for logging)
 */
export enum SecurityEventType {
  LOGIN_SUCCESS = 'LOGIN_SUCCESS',
  LOGIN_FAILED = 'LOGIN_FAILED',
  RATE_LIMIT_EXCEEDED = 'RATE_LIMIT_EXCEEDED',
}

export interface SecurityEvent {
  type: SecurityEventType;
  userId?: string;
  timestamp: number;
  ipAddress?: string;
  userAgent?: string;
  metadata?: Record<string, any>;
}

/**
 * Client-side security event logging
 */
export const logSecurityEvent = (event: SecurityEvent): void => {
  if (process.env.NODE_ENV === 'development') {
    console.log('[SECURITY EVENT]', {
      ...event,
      timestamp: new Date(event.timestamp).toISOString(),
    });
  }
};
