/**
 * CSRF Protection Implementation
 * For static export architecture - client-side token management
 * Prevents Cross-Site Request Forgery attacks
 */



/**
 * CSRF configuration
 */
export const CSRF_CONFIG = {
  TOKEN_LENGTH: 32,
  TOKEN_EXPIRY: 60 * 60, // 1 hour
  COOKIE_NAME: '__Host-csrf-token',
  HEADER_NAME: 'X-CSRF-Token',
  FORM_FIELD_NAME: '_csrf',

  // Methods that require CSRF protection
  PROTECTED_METHODS: ['POST', 'PUT', 'PATCH', 'DELETE'] as const,

  // Origins to exclude from CSRF checks (be very careful with this)
  EXCLUDED_PATHS: [
    '/api/auth/login',
    '/api/webhooks', // Webhooks typically use other auth methods
  ] as const,
} as const;

/**
 * Client-side CSRF token management
 */
export class CSRFManager {
  private token: string | null = null;
  private tokenExpiry: number | null = null;

  /**
   * Gets current CSRF token, fetching new one if expired
   */
  async getToken(): Promise<string> {
    // Check if current token is still valid
    if (this.token && this.tokenExpiry && Date.now() < this.tokenExpiry) {
      return this.token;
    }

    // Fetch new token from server
    try {
      const response = await fetch('/api/auth/csrf', {
        method: 'GET',
        credentials: 'include',
      });

      if (!response.ok) {
        throw new Error('Failed to fetch CSRF token');
      }

      const data = await response.json();
      this.token = data.token;
      this.tokenExpiry = Date.now() + (CSRF_CONFIG.TOKEN_EXPIRY * 1000);

      return this.token;
    } catch (error) {
      console.error('CSRF token fetch failed:', error);
      throw error;
    }
  }

  /**
   * Checks if a request needs CSRF protection
   */
  requiresCSRF(method: string, path: string): boolean {
    // Check if method requires CSRF
    if (!CSRF_CONFIG.PROTECTED_METHODS.includes(method as any)) {
      return false;
    }

    // Check if path is excluded
    return !CSRF_CONFIG.EXCLUDED_PATHS.some(excluded => path.startsWith(excluded));
  }
}

// Singleton instance
export const csrfManager = new CSRFManager();
