/**
 * Rate Limiting Implementation
 * Prevents brute force attacks and API abuse
 */

export interface RateLimitConfig {
  maxAttempts: number;
  windowMs: number;
  blockDurationMs?: number;
  skipSuccessfulAttempts?: boolean;
}

export interface RateLimitResult {
  allowed: boolean;
  remaining: number;
  resetTime: number;
  blocked: boolean;
}

/**
 * Client-side rate limiter (for UI feedback)
 * Server-side rate limiting should be primary defense
 */
export class ClientRateLimiter {
  private attempts: Map<string, number[]> = new Map();
  private blockedUntil: Map<string, number> = new Map();

  constructor(private config: RateLimitConfig) {}

  /**
   * Checks if action is allowed
   */
  check(key: string): RateLimitResult {
    const now = Date.now();

    // Check if currently blocked
    const blockExpiry = this.blockedUntil.get(key);
    if (blockExpiry && now < blockExpiry) {
      return {
        allowed: false,
        remaining: 0,
        resetTime: blockExpiry,
        blocked: true,
      };
    }

    // Get attempt history
    const attemptHistory = this.attempts.get(key) || [];

    // Filter attempts within window
    const windowStart = now - this.config.windowMs;
    const recentAttempts = attemptHistory.filter(time => time > windowStart);

    // Check if limit exceeded
    if (recentAttempts.length >= this.config.maxAttempts) {
      // Block the key
      const blockUntil = now + (this.config.blockDurationMs || this.config.windowMs);
      this.blockedUntil.set(key, blockUntil);

      return {
        allowed: false,
        remaining: 0,
        resetTime: blockUntil,
        blocked: true,
      };
    }

    return {
      allowed: true,
      remaining: this.config.maxAttempts - recentAttempts.length - 1,
      resetTime: windowStart + this.config.windowMs,
      blocked: false,
    };
  }

  /**
   * Records an attempt
   */
  record(key: string, success: boolean = false): void {
    // Skip recording if configured and attempt was successful
    if (this.config.skipSuccessfulAttempts && success) {
      this.reset(key);
      return;
    }

    const now = Date.now();
    const attemptHistory = this.attempts.get(key) || [];

    // Add current attempt
    attemptHistory.push(now);

    // Clean old attempts
    const windowStart = now - this.config.windowMs;
    const recentAttempts = attemptHistory.filter(time => time > windowStart);

    this.attempts.set(key, recentAttempts);
  }

  /**
   * Resets attempts for a key
   */
  reset(key: string): void {
    this.attempts.delete(key);
    this.blockedUntil.delete(key);
  }

  /**
   * Clears all data (for cleanup)
   */
  clearAll(): void {
    this.attempts.clear();
    this.blockedUntil.clear();
  }
}

/**
 * Predefined rate limit configurations
 */
export const RATE_LIMIT_CONFIGS = {
  // Login attempts
  LOGIN: {
    maxAttempts: 5,
    windowMs: 15 * 60 * 1000, // 15 minutes
    blockDurationMs: 30 * 60 * 1000, // 30 minutes
    skipSuccessfulAttempts: true,
  },

  // Password reset
  PASSWORD_RESET: {
    maxAttempts: 3,
    windowMs: 60 * 60 * 1000, // 1 hour
    blockDurationMs: 2 * 60 * 60 * 1000, // 2 hours
  },

  // API calls
  API_GENERAL: {
    maxAttempts: 100,
    windowMs: 60 * 1000, // 1 minute
    blockDurationMs: 5 * 60 * 1000, // 5 minutes
  },

  // File uploads
  UPLOAD: {
    maxAttempts: 10,
    windowMs: 60 * 1000, // 1 minute
    blockDurationMs: 10 * 60 * 1000, // 10 minutes
  },

  // Project saves
  PROJECT_SAVE: {
    maxAttempts: 20,
    windowMs: 60 * 1000, // 1 minute
  },
} as const;

/**
 * React hook for rate limiting
 */
export const useRateLimit = (config: RateLimitConfig, key: string = 'default') => {
  const limiterRef = React.useRef<ClientRateLimiter>();
  const [state, setState] = React.useState<RateLimitResult>({
    allowed: true,
    remaining: config.maxAttempts,
    resetTime: Date.now() + config.windowMs,
    blocked: false,
  });

  // Initialize limiter
  React.useEffect(() => {
    if (!limiterRef.current) {
      limiterRef.current = new ClientRateLimiter(config);
    }
  }, [config]);

  // Check rate limit
  const check = React.useCallback((): RateLimitResult => {
    if (!limiterRef.current) {
      return state;
    }

    const result = limiterRef.current.check(key);
    setState(result);
    return result;
  }, [key, state]);

  // Record attempt
  const record = React.useCallback((success: boolean = false) => {
    if (!limiterRef.current) return;

    limiterRef.current.record(key, success);

    // Update state
    const result = limiterRef.current.check(key);
    setState(result);
  }, [key]);

  // Reset limiter
  const reset = React.useCallback(() => {
    if (!limiterRef.current) return;

    limiterRef.current.reset(key);
    setState({
      allowed: true,
      remaining: config.maxAttempts,
      resetTime: Date.now() + config.windowMs,
      blocked: false,
    });
  }, [key, config]);

  return {
    ...state,
    check,
    record,
    reset,
  };
};

import React from 'react';

/**
 * Server-side rate limiting (in-memory)
 * For production, use Redis or similar distributed cache
 */
export class ServerRateLimiter {
  private attempts: Map<string, {
    count: number;
    resetTime: number;
    blocked: boolean;
  }> = new Map();

  constructor(private config: RateLimitConfig) {
    // Cleanup old entries periodically
    setInterval(() => this.cleanup(), 60000); // Every minute
  }

  /**
   * Checks rate limit for identifier
   */
  check(identifier: string): RateLimitResult {
    const now = Date.now();
    const entry = this.attempts.get(identifier);

    // No previous attempts
    if (!entry) {
      return {
        allowed: true,
        remaining: this.config.maxAttempts - 1,
        resetTime: now + this.config.windowMs,
        blocked: false,
      };
    }

    // Check if blocked
    if (entry.blocked && now < entry.resetTime) {
      return {
        allowed: false,
        remaining: 0,
        resetTime: entry.resetTime,
        blocked: true,
      };
    }

    // Reset if window expired
    if (now >= entry.resetTime) {
      this.attempts.delete(identifier);
      return {
        allowed: true,
        remaining: this.config.maxAttempts - 1,
        resetTime: now + this.config.windowMs,
        blocked: false,
      };
    }

    // Check if limit exceeded
    if (entry.count >= this.config.maxAttempts) {
      const blockUntil = now + (this.config.blockDurationMs || this.config.windowMs);

      this.attempts.set(identifier, {
        count: entry.count + 1,
        resetTime: blockUntil,
        blocked: true,
      });

      return {
        allowed: false,
        remaining: 0,
        resetTime: blockUntil,
        blocked: true,
      };
    }

    return {
      allowed: true,
      remaining: this.config.maxAttempts - entry.count - 1,
      resetTime: entry.resetTime,
      blocked: false,
    };
  }

  /**
   * Records an attempt
   */
  record(identifier: string): void {
    const now = Date.now();
    const entry = this.attempts.get(identifier);

    if (!entry || now >= entry.resetTime) {
      this.attempts.set(identifier, {
        count: 1,
        resetTime: now + this.config.windowMs,
        blocked: false,
      });
    } else {
      entry.count++;
    }
  }

  /**
   * Resets rate limit for identifier
   */
  reset(identifier: string): void {
    this.attempts.delete(identifier);
  }

  /**
   * Cleanup expired entries
   */
  private cleanup(): void {
    const now = Date.now();
    const toDelete: string[] = [];

    for (const [key, entry] of this.attempts.entries()) {
      if (now >= entry.resetTime) {
        toDelete.push(key);
      }
    }

    toDelete.forEach(key => this.attempts.delete(key));
  }

  /**
   * Gets current stats
   */
  getStats(): {
    totalEntries: number;
    blockedIdentifiers: number;
    } {
    let blocked = 0;
    for (const entry of this.attempts.values()) {
      if (entry.blocked) blocked++;
    }

    return {
      totalEntries: this.attempts.size,
      blockedIdentifiers: blocked,
    };
  }
}

/**
 * Creates identifier from request
 */
export const getIdentifier = (
  request: Request,
  useIP: boolean = true,
  useUserAgent: boolean = true
): string => {
  const parts: string[] = [];

  if (useIP) {
    // Get IP from various headers (depends on proxy setup)
    const ip =
      request.headers.get('x-forwarded-for')?.split(',')[0] ||
      request.headers.get('x-real-ip') ||
      request.headers.get('cf-connecting-ip') || // Cloudflare
      'unknown';

    parts.push(ip);
  }

  if (useUserAgent) {
    const userAgent = request.headers.get('user-agent') || 'unknown';
    parts.push(userAgent);
  }

  return parts.join('|');
};

/**
 * Rate limit middleware helper for API routes
 */
export const withRateLimit = (
  config: RateLimitConfig,
  handler: (request: Request) => Promise<Response>,
  getKey?: (request: Request) => string
) => {
  const limiter = new ServerRateLimiter(config);

  return async (request: Request): Promise<Response> => {
    const identifier = getKey ? getKey(request) : getIdentifier(request);
    const result = limiter.check(identifier);

    if (!result.allowed) {
      const retryAfter = Math.ceil((result.resetTime - Date.now()) / 1000);

      return new Response(
        JSON.stringify({
          error: 'Too many requests',
          retryAfter,
        }),
        {
          status: 429,
          headers: {
            'Content-Type': 'application/json',
            'Retry-After': retryAfter.toString(),
            'X-RateLimit-Limit': config.maxAttempts.toString(),
            'X-RateLimit-Remaining': '0',
            'X-RateLimit-Reset': result.resetTime.toString(),
          },
        }
      );
    }

    limiter.record(identifier);

    const response = await handler(request);

    // Add rate limit headers to successful responses
    response.headers.set('X-RateLimit-Limit', config.maxAttempts.toString());
    response.headers.set('X-RateLimit-Remaining', result.remaining.toString());
    response.headers.set('X-RateLimit-Reset', result.resetTime.toString());

    return response;
  };
};

/**
 * Distributed rate limiter using Redis (for production)
 * Requires Redis client
 */
export class RedisRateLimiter {
  constructor(
    private redis: any, // Redis client
    private config: RateLimitConfig
  ) {}

  async check(identifier: string): Promise<RateLimitResult> {
    const key = `ratelimit:${identifier}`;
    const now = Date.now();

    const multi = this.redis.multi();
    multi.zremrangebyscore(key, 0, now - this.config.windowMs);
    multi.zcard(key);
    multi.zadd(key, now, `${now}`);
    multi.expire(key, Math.ceil(this.config.windowMs / 1000));

    const results = await multi.exec();
    const count = results[1][1];

    if (count > this.config.maxAttempts) {
      return {
        allowed: false,
        remaining: 0,
        resetTime: now + this.config.windowMs,
        blocked: true,
      };
    }

    return {
      allowed: true,
      remaining: this.config.maxAttempts - count,
      resetTime: now + this.config.windowMs,
      blocked: false,
    };
  }

  async reset(identifier: string): Promise<void> {
    const key = `ratelimit:${identifier}`;
    await this.redis.del(key);
  }
}