/**
 * Secure Client-Side Storage Utilities
 * For static export architecture
 *
 * SECURITY PRINCIPLES:
 * - Never store sensitive data (tokens, passwords) in localStorage/sessionStorage
 * - Use HttpOnly cookies for authentication (managed by backend)
 * - Only store non-sensitive UI state and preferences
 * - Sanitize all data before storage
 * - Validate all data when reading
 */

/**
 * Storage types
 */
export enum StorageType {
  LOCAL = 'local',
  SESSION = 'session',
}

/**
 * Allowed storage keys (whitelist approach)
 * Only these keys can be stored
 */
export const ALLOWED_STORAGE_KEYS = {
  // UI Preferences
  UI_THEME: 'ui_theme',
  UI_LANGUAGE: 'ui_language',
  UI_SIDEBAR_COLLAPSED: 'ui_sidebar_collapsed',
  UI_ZOOM_LEVEL: 'ui_zoom_level',

  // User Preferences
  USER_PREFERENCES: 'user_preferences',

  // Application State (non-sensitive)
  LAST_VISITED_PAGE: 'last_visited_page',
  ONBOARDING_COMPLETED: 'onboarding_completed',

  // CSRF Token (non-sensitive, for convenience)
  CSRF_TOKEN: 'csrf_token',
  CSRF_TOKEN_EXPIRY: 'csrf_token_expiry',
} as const;

export type AllowedStorageKey = typeof ALLOWED_STORAGE_KEYS[keyof typeof ALLOWED_STORAGE_KEYS];

/**
 * Forbidden patterns - never allow these to be stored
 */
const FORBIDDEN_PATTERNS = [
  /token/i,
  /password/i,
  /secret/i,
  /key/i,
  /auth/i,
  /session/i,
  /jwt/i,
  /credential/i,
];

/**
 * Check if a key is safe to store
 */
const isSafeKey = (key: string): boolean => {
  // Check if key is in whitelist
  const isWhitelisted = Object.values(ALLOWED_STORAGE_KEYS).includes(key as AllowedStorageKey);

  // Check if key matches forbidden patterns
  const isForbidden = FORBIDDEN_PATTERNS.some(pattern => pattern.test(key));

  return isWhitelisted && !isForbidden;
};

/**
 * Sanitize value before storage
 */
const sanitizeValue = (value: any): string => {
  if (typeof value === 'string') {
    // Remove any script tags
    return value.replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '');
  }
  return JSON.stringify(value);
};

/**
 * Parse value from storage
 */
const parseValue = <T>(value: string | null): T | null => {
  if (!value) return null;

  try {
    return JSON.parse(value) as T;
  } catch {
    // If not valid JSON, return as string
    return value as unknown as T;
  }
};

/**
 * Get storage instance
 */
const getStorage = (type: StorageType): Storage | null => {
  if (typeof window === 'undefined') return null;

  try {
    return type === StorageType.LOCAL ? window.localStorage : window.sessionStorage;
  } catch {
    // Storage might be disabled
    return null;
  }
};

/**
 * Secure storage interface
 */
export class SecureStorage {
  private storage: Storage | null;
  private storageType: StorageType;

  constructor(type: StorageType = StorageType.LOCAL) {
    this.storageType = type;
    this.storage = getStorage(type);
  }

  /**
   * Store a value securely
   */
  set<T>(key: AllowedStorageKey, value: T): boolean {
    if (!this.storage) {
      console.warn('[SecureStorage] Storage not available');
      return false;
    }

    if (!isSafeKey(key)) {
      console.error(`[SecureStorage] Key "${key}" is not allowed for storage`);
      return false;
    }

    try {
      const sanitized = sanitizeValue(value);
      this.storage.setItem(key, sanitized);
      return true;
    } catch (error) {
      console.error('[SecureStorage] Failed to store value:', error);
      return false;
    }
  }

  /**
   * Retrieve a value
   */
  get<T>(key: AllowedStorageKey): T | null {
    if (!this.storage) {
      return null;
    }

    if (!isSafeKey(key)) {
      console.error(`[SecureStorage] Key "${key}" is not allowed`);
      return null;
    }

    try {
      const value = this.storage.getItem(key);
      return parseValue<T>(value);
    } catch (error) {
      console.error('[SecureStorage] Failed to retrieve value:', error);
      return null;
    }
  }

  /**
   * Remove a value
   */
  remove(key: AllowedStorageKey): boolean {
    if (!this.storage) {
      return false;
    }

    try {
      this.storage.removeItem(key);
      return true;
    } catch (error) {
      console.error('[SecureStorage] Failed to remove value:', error);
      return false;
    }
  }

  /**
   * Clear all stored values
   */
  clear(): boolean {
    if (!this.storage) {
      return false;
    }

    try {
      this.storage.clear();
      return true;
    } catch (error) {
      console.error('[SecureStorage] Failed to clear storage:', error);
      return false;
    }
  }

  /**
   * Check if a key exists
   */
  has(key: AllowedStorageKey): boolean {
    if (!this.storage) {
      return false;
    }

    return this.storage.getItem(key) !== null;
  }

  /**
   * Get all keys
   */
  keys(): string[] {
    if (!this.storage) {
      return [];
    }

    try {
      return Object.keys(this.storage);
    } catch {
      return [];
    }
  }
}

/**
 * Default storage instances
 */
export const secureLocalStorage = new SecureStorage(StorageType.LOCAL);
export const secureSessionStorage = new SecureStorage(StorageType.SESSION);

/**
 * React hook for secure storage
 */
export const useSecureStorage = <T>(
  key: AllowedStorageKey,
  initialValue: T,
  type: StorageType = StorageType.LOCAL
): [T, (value: T) => void, () => void] => {
  const storage = type === StorageType.LOCAL ? secureLocalStorage : secureSessionStorage;

  // Initialize from storage or use initial value
  const [storedValue, setStoredValue] = React.useState<T>(() => {
    const item = storage.get<T>(key);
    return item !== null ? item : initialValue;
  });

  // Update storage when value changes
  const setValue = (value: T) => {
    setStoredValue(value);
    storage.set(key, value);
  };

  // Clear value
  const clearValue = () => {
    setStoredValue(initialValue);
    storage.remove(key);
  };

  return [storedValue, setValue, clearValue];
};

// Import React for hook
import React from 'react';

/**
 * CSRF Token storage helpers
 * CSRF tokens are safe to store client-side (they're not authentication tokens)
 */
export const csrfTokenStorage = {
  set: (token: string, expiresAt: number): boolean => {
    const success1 = secureLocalStorage.set(ALLOWED_STORAGE_KEYS.CSRF_TOKEN, token);
    const success2 = secureLocalStorage.set(ALLOWED_STORAGE_KEYS.CSRF_TOKEN_EXPIRY, expiresAt);
    return success1 && success2;
  },

  get: (): { token: string; expiresAt: number } | null => {
    const token = secureLocalStorage.get<string>(ALLOWED_STORAGE_KEYS.CSRF_TOKEN);
    const expiresAt = secureLocalStorage.get<number>(ALLOWED_STORAGE_KEYS.CSRF_TOKEN_EXPIRY);

    if (!token || !expiresAt) return null;

    // Check if expired
    if (Date.now() > expiresAt) {
      csrfTokenStorage.clear();
      return null;
    }

    return { token, expiresAt };
  },

  clear: (): void => {
    secureLocalStorage.remove(ALLOWED_STORAGE_KEYS.CSRF_TOKEN);
    secureLocalStorage.remove(ALLOWED_STORAGE_KEYS.CSRF_TOKEN_EXPIRY);
  },
};