/**
 * Security Validation Utilities
 * Input validation and sanitization
 */

export interface ValidationResult {
  valid: boolean;
  error?: string;
  strength?: 'weak' | 'fair' | 'good' | 'strong' | 'excellent';
}

/**
 * Password strength checker
 */
export const calculatePasswordStrength = (password: string): {
  score: number; // 0-100
  strength: 'weak' | 'fair' | 'good' | 'strong' | 'excellent';
  feedback: string[];
} => {
  let score = 0;
  const feedback: string[] = [];

  // Length check (0-30 points)
  if (password.length >= 12) {
    score += 30;
  } else if (password.length >= 10) {
    score += 20;
    feedback.push('Use at least 12 characters for better security');
  } else if (password.length >= 8) {
    score += 10;
    feedback.push('Password is too short - use at least 12 characters');
  } else {
    feedback.push('Password must be at least 8 characters');
  }

  // Character variety (0-40 points)
  const hasLower = /[a-z]/.test(password);
  const hasUpper = /[A-Z]/.test(password);
  const hasNumber = /[0-9]/.test(password);
  const hasSpecial = /[^a-zA-Z0-9]/.test(password);

  const varietyCount = [hasLower, hasUpper, hasNumber, hasSpecial].filter(Boolean).length;

  switch (varietyCount) {
  case 4:
    score += 40;
    break;
  case 3:
    score += 25;
    feedback.push('Add special characters for stronger password');
    break;
  case 2:
    score += 15;
    feedback.push('Add uppercase, numbers, and special characters');
    break;
  case 1:
    score += 5;
    feedback.push('Use a mix of uppercase, lowercase, numbers, and special characters');
    break;
  }

  // Entropy check (0-20 points)
  const uniqueChars = new Set(password).size;
  const entropyScore = Math.min(20, (uniqueChars / password.length) * 20);
  score += entropyScore;

  if (entropyScore < 10) {
    feedback.push('Avoid repeated characters');
  }

  // Pattern detection (negative points)
  const commonPatterns = [
    /12345/, /qwerty/, /password/, /admin/, /letmein/, /welcome/,
    /(.)\1{2,}/, // repeated characters
    /(abc|def|ghi|jkl|mno|pqr|stu|vwx|yz)/i, // sequential letters
    /(012|123|234|345|456|567|678|789)/  // sequential numbers
  ];

  for (const pattern of commonPatterns) {
    if (pattern.test(password.toLowerCase())) {
      score -= 15;
      feedback.push('Avoid common patterns and sequences');
      break;
    }
  }

  // Dictionary words check (simplified)
  const commonWords = ['password', 'admin', 'user', 'login', 'welcome', 'test'];
  if (commonWords.some(word => password.toLowerCase().includes(word))) {
    score -= 10;
    feedback.push('Avoid common words');
  }

  // Ensure score is 0-100
  score = Math.max(0, Math.min(100, score));

  // Determine strength category
  let strength: 'weak' | 'fair' | 'good' | 'strong' | 'excellent';
  if (score >= 80) strength = 'excellent';
  else if (score >= 60) strength = 'strong';
  else if (score >= 40) strength = 'good';
  else if (score >= 20) strength = 'fair';
  else strength = 'weak';

  return { score, strength, feedback };
};

/**
 * Comprehensive password validation
 */
export const validatePassword = (password: string): ValidationResult => {
  // Minimum length
  if (password.length < 6) {
    return {
      valid: false,
      error: 'Password must be at least 8 characters long'
    };
  }

  // Maximum length (prevent DoS)
  if (password.length > 128) {
    return {
      valid: false,
      error: 'Password must not exceed 128 characters'
    };
  }

  // Check strength
  const { strength } = calculatePasswordStrength(password);

  if (strength === 'weak') {
    return {
      valid: false,
      error: 'Password is too weak. Please choose a stronger password.',
      strength
    };
  }

  return {
    valid: true,
    strength
  };
};

/**
 * Email validation (RFC 5322 compliant)
 */
export const validateEmail = (email: string): ValidationResult => {
  if (!email) {
    return {
      valid: false,
      error: 'Email is required'
    };
  }

  // Length check
  if (email.length > 254) {
    return {
      valid: false,
      error: 'Email is too long'
    };
  }

  // RFC 5322 simplified regex
  const emailRegex = /^[a-zA-Z0-9.!#$%&'*+\/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;

  if (!emailRegex.test(email)) {
    return {
      valid: false,
      error: 'Invalid email format'
    };
  }

  // Additional checks
  const [localPart, domain] = email.split('@');

  if (localPart.length > 64) {
    return {
      valid: false,
      error: 'Email local part is too long'
    };
  }

  // Check for disposable email domains (optional)
  const disposableDomains = ['tempmail.com', '10minutemail.com', 'guerrillamail.com'];
  if (disposableDomains.some(d => domain.toLowerCase().includes(d))) {
    return {
      valid: false,
      error: 'Disposable email addresses are not allowed'
    };
  }

  return { valid: true };
};

/**
 * Username validation
 */
export const validateUsername = (username: string): ValidationResult => {
  if (!username) {
    return {
      valid: false,
      error: 'Username is required'
    };
  }

  if (username.length < 3) {
    return {
      valid: false,
      error: 'Username must be at least 3 characters'
    };
  }

  if (username.length > 30) {
    return {
      valid: false,
      error: 'Username must not exceed 30 characters'
    };
  }

  // Alphanumeric with underscores and hyphens
  if (!/^[a-zA-Z0-9_-]+$/.test(username)) {
    return {
      valid: false,
      error: 'Username can only contain letters, numbers, underscores, and hyphens'
    };
  }

  // Must start with letter
  if (!/^[a-zA-Z]/.test(username)) {
    return {
      valid: false,
      error: 'Username must start with a letter'
    };
  }

  return { valid: true };
};

/**
 * URL validation
 */
export const validateURL = (url: string, allowedProtocols: string[] = ['http', 'https']): ValidationResult => {
  if (!url) {
    return {
      valid: false,
      error: 'URL is required'
    };
  }

  try {
    const parsed = new URL(url);

    // Check protocol
    const protocol = parsed.protocol.replace(':', '');
    if (!allowedProtocols.includes(protocol)) {
      return {
        valid: false,
        error: `Only ${allowedProtocols.join(', ')} protocols are allowed`
      };
    }

    // Check for localhost/internal IPs in production
    if (process.env.NODE_ENV === 'production') {
      const hostname = parsed.hostname;
      if (
        hostname === 'localhost' ||
        hostname === '127.0.0.1' ||
        hostname.startsWith('192.168.') ||
        hostname.startsWith('10.') ||
        hostname.match(/^172\.(1[6-9]|2[0-9]|3[0-1])\./)
      ) {
        return {
          valid: false,
          error: 'Internal URLs are not allowed'
        };
      }
    }

    return { valid: true };
  } catch {
    return {
      valid: false,
      error: 'Invalid URL format'
    };
  }
};

/**
 * Input sanitization for XSS prevention
 */
export const sanitizeInput = (input: string): string => {
  if (!input) return '';

  return input
    .replace(/[<>]/g, '') // Remove angle brackets
    .replace(/javascript:/gi, '') // Remove javascript: protocol
    .replace(/on\w+=/gi, '') // Remove event handlers
    .trim();
};

/**
 * SQL injection prevention (basic)
 * Note: Use parameterized queries on backend for proper protection
 */
export const sanitizeSQLInput = (input: string): string => {
  if (!input) return '';

  return input
    .replace(/['";\\]/g, '') // Remove SQL special characters
    .replace(/--/g, '') // Remove SQL comments
    .replace(/\/\*/g, '') // Remove SQL block comments
    .trim();
};

/**
 * File name sanitization
 */
export const sanitizeFileName = (fileName: string): string => {
  if (!fileName) return '';

  return fileName
    .replace(/[^a-zA-Z0-9._-]/g, '_') // Replace special chars
    .replace(/\.\./g, '_') // Prevent directory traversal
    .replace(/^\./, '_') // Remove leading dot
    .substring(0, 255); // Limit length
};

/**
 * Phone number validation (international format)
 */
export const validatePhoneNumber = (phone: string): ValidationResult => {
  if (!phone) {
    return {
      valid: false,
      error: 'Phone number is required'
    };
  }

  // Remove spaces, hyphens, and parentheses
  const cleaned = phone.replace(/[\s\-()]/g, '');

  // International format: optional +, 7-15 digits
  if (!/^\+?[1-9]\d{6,14}$/.test(cleaned)) {
    return {
      valid: false,
      error: 'Invalid phone number format'
    };
  }

  return { valid: true };
};

/**
 * Credit card validation (basic Luhn algorithm)
 */
export const validateCreditCard = (cardNumber: string): ValidationResult => {
  const cleaned = cardNumber.replace(/\s/g, '');

  if (!/^\d{13,19}$/.test(cleaned)) {
    return {
      valid: false,
      error: 'Invalid card number length'
    };
  }

  // Luhn algorithm
  let sum = 0;
  let isEven = false;

  for (let i = cleaned.length - 1; i >= 0; i--) {
    let digit = parseInt(cleaned[i], 10);

    if (isEven) {
      digit *= 2;
      if (digit > 9) {
        digit -= 9;
      }
    }

    sum += digit;
    isEven = !isEven;
  }

  if (sum % 10 !== 0) {
    return {
      valid: false,
      error: 'Invalid card number'
    };
  }

  return { valid: true };
};

/**
 * IP address validation
 */
export const validateIPAddress = (ip: string): ValidationResult => {
  // IPv4
  const ipv4Regex = /^(\d{1,3}\.){3}\d{1,3}$/;
  if (ipv4Regex.test(ip)) {
    const parts = ip.split('.');
    if (parts.every(part => parseInt(part, 10) <= 255)) {
      return { valid: true };
    }
  }

  // IPv6 (simplified)
  const ipv6Regex = /^([0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}$/;
  if (ipv6Regex.test(ip)) {
    return { valid: true };
  }

  return {
    valid: false,
    error: 'Invalid IP address'
  };
};

/**
 * Date validation
 */
export const validateDate = (
  dateString: string,
  minDate?: Date,
  maxDate?: Date
): ValidationResult => {
  const date = new Date(dateString);

  if (isNaN(date.getTime())) {
    return {
      valid: false,
      error: 'Invalid date format'
    };
  }

  if (minDate && date < minDate) {
    return {
      valid: false,
      error: `Date must be after ${minDate.toISOString().split('T')[0]}`
    };
  }

  if (maxDate && date > maxDate) {
    return {
      valid: false,
      error: `Date must be before ${maxDate.toISOString().split('T')[0]}`
    };
  }

  return { valid: true };
};

/**
 * Comprehensive form validation
 */
export interface FormField {
  value: string;
  type: 'email' | 'password' | 'username' | 'url' | 'phone' | 'text';
  required?: boolean;
  minLength?: number;
  maxLength?: number;
  pattern?: RegExp;
}

export const validateForm = (
  fields: Record<string, FormField>
): Record<string, ValidationResult> => {
  const results: Record<string, ValidationResult> = {};

  for (const [name, field] of Object.entries(fields)) {
    // Required check
    if (field.required && !field.value) {
      results[name] = {
        valid: false,
        error: 'This field is required'
      };
      continue;
    }

    // Skip validation for optional empty fields
    if (!field.required && !field.value) {
      results[name] = { valid: true };
      continue;
    }

    // Type-specific validation
    switch (field.type) {
    case 'email':
      results[name] = validateEmail(field.value);
      break;
    case 'password':
      results[name] = validatePassword(field.value);
      break;
    case 'username':
      results[name] = validateUsername(field.value);
      break;
    case 'url':
      results[name] = validateURL(field.value);
      break;
    case 'phone':
      results[name] = validatePhoneNumber(field.value);
      break;
    default:
      // Text validation
      if (field.minLength && field.value.length < field.minLength) {
        results[name] = {
          valid: false,
          error: `Must be at least ${field.minLength} characters`
        };
      } else if (field.maxLength && field.value.length > field.maxLength) {
        results[name] = {
          valid: false,
          error: `Must not exceed ${field.maxLength} characters`
        };
      } else if (field.pattern && !field.pattern.test(field.value)) {
        results[name] = {
          valid: false,
          error: 'Invalid format'
        };
      } else {
        results[name] = { valid: true };
      }
    }
  }

  return results;
};
