import React, { Component, ReactNode } from 'react';
import { EuiEmptyPrompt, EuiButton, EuiCallOut, EuiSpacer, EuiCode } from '@elastic/eui';

interface Props {
  children: ReactNode;
  fallback?: ReactNode;
  onReset?: () => void;
}

interface State {
  hasError: boolean;
  error: Error | null;
  errorInfo: React.ErrorInfo | null;
}

/**
 * ErrorBoundary - Catches React errors and displays fallback UI
 *
 * Prevents entire app crashes by catching errors in child components
 * and displaying a user-friendly error message with recovery options.
 *
 */
export class ErrorBoundary extends Component<Props, State> {
  constructor(props: Props) {
    super(props);
    this.state = {
      hasError: false,
      error: null,
      errorInfo: null
    };
  }

  static getDerivedStateFromError(error: Error): Partial<State> {
    // Update state so the next render will show the fallback UI
    return { hasError: true, error };
  }

  componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
    // Log error details for debugging
    console.error('ErrorBoundary caught an error:', error, errorInfo);

    // Update state with error info
    this.setState({
      error,
      errorInfo
    });

    // TODO: Send to error reporting service (e.g., Sentry, LogRocket)
  }

  handleReset = () => {
    this.setState({
      hasError: false,
      error: null,
      errorInfo: null
    });

    // Call optional reset callback
    if (this.props.onReset) {
      this.props.onReset();
    }
  };

  render() {
    if (this.state.hasError) {
      // Use custom fallback if provided
      if (this.props.fallback) {
        return this.props.fallback;
      }

      // Default error UI
      return (
        <div style={{ padding: '24px', maxWidth: '800px', margin: '0 auto' }}>
          <EuiEmptyPrompt
            iconType="alert"
            iconColor="danger"
            title={<h2>Something went wrong</h2>}
            body={
              <>
                <p>
                  An unexpected error occurred. You can try reloading the page or returning to the previous screen.
                </p>

                {this.state.error && (
                  <>
                    <EuiSpacer size="m" />
                    <EuiCallOut
                      title="Error Details"
                      color="danger"
                      iconType="alert"
                    >
                      <p>
                        <strong>Error:</strong> {this.state.error.message}
                      </p>
                      {process.env.NODE_ENV === 'development' && this.state.errorInfo && (
                        <>
                          <EuiSpacer size="s" />
                          <details style={{ whiteSpace: 'pre-wrap', fontSize: '12px' }}>
                            <summary style={{ cursor: 'pointer', fontWeight: 'bold' }}>
                              Component Stack (Development Only)
                            </summary>
                            <EuiCode style={{ display: 'block', marginTop: '8px', padding: '8px' }}>
                              {this.state.errorInfo.componentStack}
                            </EuiCode>
                          </details>
                        </>
                      )}
                    </EuiCallOut>
                  </>
                )}
              </>
            }
            actions={[
              <EuiButton
                key={'0'}
                color="primary"
                fill
                onClick={this.handleReset}
              >
                Try Again
              </EuiButton>,
              <EuiButton
                key={'1'}
                color="primary"
                onClick={() => window.location.reload()}
              >
                Reload Page
              </EuiButton>
            ]}
          />
        </div>
      );
    }

    return this.props.children;
  }
}

/**
 * withErrorBoundary - HOC to wrap components with error boundary
 *
 * @example
 * export default withErrorBoundary(MyComponent);
 */
export function withErrorBoundary<P extends object>(
  Component: React.ComponentType<P>,
  fallback?: ReactNode
) {
  return function WithErrorBoundary(props: P) {
    return (
      <ErrorBoundary fallback={fallback}>
        <Component {...props} />
      </ErrorBoundary>
    );
  };
}
