

import { useState, useEffect, useCallback, useRef } from 'react';
import { APIError } from '@/lib/api';

/**
 * Configuration options for useApiData
 */
export interface UseApiDataOptions<T> {
  /** Auto-fetch data on mount */
  autoFetch?: boolean;
  /** Refetch interval in milliseconds (0 = no polling) */
  refetchInterval?: number;
  /** Initial data value */
  initialData?: T;
  /** Callback when data is successfully fetched */
  onSuccess?: (data: T) => void;
  /** Callback when an error occurs */
  onError?: (error: APIError) => void;
  /** Enable optimistic updates */
  enableOptimisticUpdates?: boolean;
}

/**
 * Hook state shape
 */
interface UseApiDataState<T> {
  data: T;
  loading: boolean;
  error: APIError | null;
  isFetched: boolean;
}

/**
 * Fetch function type - async function that returns data
 */
export type FetchFunction<T> = () => Promise<T>;

export function useApiData<T>(
  fetchFn: FetchFunction<T>,
  options: UseApiDataOptions<T> = {}
) {
  const {
    autoFetch = true,
    refetchInterval = 0,
    initialData = null as T,
    onSuccess,
    onError,
    enableOptimisticUpdates = false,
  } = options;

  // State
  const [state, setState] = useState<UseApiDataState<T>>({
    data: initialData,
    loading: false,
    error: null,
    isFetched: false,
  });

  // Refs for cleanup and stable function references
  const abortControllerRef = useRef<AbortController | null>(null);
  const isMountedRef = useRef(true);
  const fetchFnRef = useRef(fetchFn);
  const onSuccessRef = useRef(onSuccess);
  const onErrorRef = useRef(onError);

  // Update refs when props change (doesn't cause re-render)
  useEffect(() => {
    fetchFnRef.current = fetchFn;
    onSuccessRef.current = onSuccess;
    onErrorRef.current = onError;
  }, [fetchFn, onSuccess, onError]);

  /**
   * Fetch data from the API
   * Uses refs to avoid recreating this function unnecessarily
   */
  const fetchData = useCallback(async () => {
    // Cancel any pending requests
    abortControllerRef.current?.abort();
    abortControllerRef.current = new AbortController();

    setState(prev => ({ ...prev, loading: true, error: null }));

    try {
      const data = await fetchFnRef.current();

      // Only update state if component is still mounted
      if (isMountedRef.current) {
        setState({
          data,
          loading: false,
          error: null,
          isFetched: true,
        });

        onSuccessRef.current?.(data);
      }
    } catch (err) {
      if (isMountedRef.current) {
        const error = err instanceof APIError
          ? err
          : new APIError(err instanceof Error ? err.message : 'Failed to fetch data');

        setState(prev => ({
          ...prev,
          loading: false,
          error,
          isFetched: true,
        }));

        onErrorRef.current?.(error);
      }
    }
  }, []); // Empty deps - stable function reference

  /**
   * Manually update the data (for optimistic updates)
   */
  const setData = useCallback((newData: T | ((prev: T) => T)) => {
    setState(prev => ({
      ...prev,
      data: typeof newData === 'function'
        ? (newData as (prev: T) => T)(prev.data)
        : newData,
    }));
  }, []);

  /**
   * Add an item to the data array (for optimistic updates on lists)
   * Only works if T is an array type
   */
  const addItem = useCallback(<Item,>(item: Item) => {
    if (enableOptimisticUpdates) {
      setState(prev => ({
        ...prev,
        data: Array.isArray(prev.data)
          ? [...prev.data, item] as unknown as T
          : prev.data,
      }));
    }
  }, [enableOptimisticUpdates]);

  /**
   * Remove an item from the data array by ID (for optimistic updates on lists)
   * Only works if T is an array type
   */
  const removeItem = useCallback((id: string) => {
    if (enableOptimisticUpdates) {
      setState(prev => ({
        ...prev,
        data: Array.isArray(prev.data)
          ? prev.data.filter((item: unknown) =>
            (item as { id?: string }).id !== id
          ) as unknown as T
          : prev.data,
      }));
    }
  }, [enableOptimisticUpdates]);

  /**
   * Update an item in the data array (for optimistic updates on lists)
   * Only works if T is an array type
   */
  const updateItem = useCallback(<Item extends { id: string }>(id: string, updater: Partial<Item> | ((item: Item) => Item)) => {
    if (enableOptimisticUpdates) {
      setState(prev => ({
        ...prev,
        data: Array.isArray(prev.data)
          ? prev.data.map((item: unknown) => {
            const typedItem = item as Item;
            if (typedItem.id === id) {
              return typeof updater === 'function'
                ? updater(typedItem)
                : { ...typedItem, ...updater };
            }
            return item;
          }) as unknown as T
          : prev.data,
      }));
    }
  }, [enableOptimisticUpdates]);

  /**
   * Clear error state
   */
  const clearError = useCallback(() => {
    setState(prev => ({ ...prev, error: null }));
  }, []);

  /**
   * Reset to initial state
   */
  const reset = useCallback(() => {
    setState({
      data: initialData,
      loading: false,
      error: null,
      isFetched: false,
    });
  }, [initialData]);

  // Auto-fetch on mount if enabled
  useEffect(() => {
    if (autoFetch) {
      fetchData();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [autoFetch]); // Only run when autoFetch changes, not when fetchData changes

  // Setup polling if interval is specified
  useEffect(() => {
    if (refetchInterval > 0) {
      const intervalId = setInterval(() => fetchData(), refetchInterval);
      return () => clearInterval(intervalId);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [refetchInterval]); // Only run when refetchInterval changes, not when fetchData changes

  // Set mounted state and cleanup on unmount
  useEffect(() => {
    isMountedRef.current = true;
    return () => {
      isMountedRef.current = false;
      abortControllerRef.current?.abort();
    };
  }, []);

  return {
    // State
    data: state.data,
    loading: state.loading,
    error: state.error,
    isFetched: state.isFetched,

    // Actions
    refetch: fetchData,
    setData,
    addItem,
    removeItem,
    updateItem,
    clearError,
    reset,
  };
}

export function useApiSingleData<T>(
  fetchFn: (id: string) => Promise<T>,
  id: string | null,
  options: Omit<UseApiDataOptions<T>, 'refetchInterval'> & { refetchInterval?: number } = {}
) {
  const { autoFetch = true, refetchInterval = 0, ...restOptions } = options;

  // Create a wrapped fetch function that uses the ID
  const wrappedFetchFn = useCallback(async () => {
    if (!id) {
      throw new Error('ID is required to fetch data');
    }
    return fetchFn(id);
  }, [fetchFn, id]);

  // Use the main hook with conditional auto-fetch
  const result = useApiData(wrappedFetchFn, {
    ...restOptions,
    autoFetch: autoFetch && id !== null,
    refetchInterval: id !== null ? refetchInterval : 0,
  });

  return result;
}

/**
 * Type helper for creating optimistic update options
 */
export interface OptimisticUpdateOptions {
  enableOptimisticUpdates: true;
}
