import { useState, useCallback, useEffect } from 'react';
import { DragItem, TouchDragPreview, AccessPointType } from '@/app/components/pdf-editor/types';
import AccessPointMarkerPoint from '@/app/types/AccessPointMarker';
import {
  screenToMapCoordinates,
  getCurrentImageDimensions,
  pixelToRelative,
  isTypeAtLimit,
  generateUniqueId
} from '@/lib/nodeScaling';
import { TOUCH, TIMING } from '@/lib/uiConstants';

interface UseDragAndDropProps {
  accessPoints: AccessPointMarkerPoint[];
  setAccessPoints: React.Dispatch<React.SetStateAction<AccessPointMarkerPoint[]>>;
  currentPage: number;
  allowNodeMovement: boolean;
  accessPointTypes: AccessPointType[];
  generateDeviceDetails: (type: string) => Promise<any>;
  mapContainerRef: React.RefObject<HTMLDivElement>;
  mapImageRef: React.RefObject<HTMLImageElement>;
  state: any;
}

export const useDragAndDrop = ({
  accessPoints,
  setAccessPoints,
  currentPage,
  allowNodeMovement,
  accessPointTypes,
  generateDeviceDetails,
  mapContainerRef,
  mapImageRef,
  state
}: UseDragAndDropProps) => {
  const [draggedItem, setDraggedItem] = useState<DragItem | null>(null);
  const [touchDragItem, setTouchDragItem] = useState<DragItem | null>(null);
  const [isDraggingTouch, setIsDraggingTouch] = useState(false);
  const [touchDragPreview, setTouchDragPreview] = useState<TouchDragPreview>({
    x: 0,
    y: 0,
    visible: false
  });
  const [dragOffset, setDragOffset] = useState({ x: 0, y: 0 });
  const [limitMessage, setLimitMessage] = useState('');

  // Handle drag start for access point types
  const handleDragStart = useCallback((e: React.DragEvent, type: string) => {
    if (isTypeAtLimit(type, accessPoints)) {
      e.preventDefault();
      setLimitMessage('Limit reached (1 max per map)');
      setTimeout(() => setLimitMessage(''), TIMING.LIMIT_MESSAGE_DURATION);
      return;
    }
    setDragOffset({ x: 0, y: 0 });
    setDraggedItem({ type, isNew: true });
    e.dataTransfer.effectAllowed = 'copy';
  }, [accessPoints]);

  // Handle touch start for access point types (mobile)
  const handleTouchStartType = useCallback((e: React.TouchEvent, type: string) => {
    if (isTypeAtLimit(type, accessPoints)) {
      e.preventDefault();
      setLimitMessage('Limit reached (1 max per map)');
      setTimeout(() => setLimitMessage(''), TIMING.LIMIT_MESSAGE_DURATION);
      return;
    }
    e.preventDefault();
    setDragOffset({ x: 0, y: 0 });
    setTouchDragItem({ type, isNew: true });
    setIsDraggingTouch(true);
    const touch = e.touches[0];
    setTouchDragPreview({
      x: touch.clientX + TOUCH.PREVIEW_OFFSET_X,
      y: touch.clientY + TOUCH.PREVIEW_OFFSET_Y,
      visible: true
    });
  }, [accessPoints]);

  // Handle drag start for existing access points
  const handleAccessPointDragStart = useCallback((e: React.DragEvent, accessPoint: AccessPointMarkerPoint) => {
    if (!allowNodeMovement) {
      e.preventDefault();
      return;
    }

    if (isDraggingTouch) {
      e.preventDefault();
      return;
    }

    const elementRect = (e.target as HTMLElement).getBoundingClientRect();
    const offsetX = e.clientX - elementRect.left - (elementRect.width / 2);
    const offsetY = e.clientY - elementRect.top - (elementRect.height / 2);

    setDragOffset({ x: offsetX, y: offsetY });
    setDraggedItem({ ...accessPoint, isNew: false } as any);

    e.dataTransfer.effectAllowed = 'move';
  }, [allowNodeMovement, isDraggingTouch]);

  // Handle touch start for existing access points (mobile)
  const handleTouchStartAccessPoint = useCallback((e: React.TouchEvent, accessPoint: AccessPointMarkerPoint) => {
    if (!allowNodeMovement) {
      e.preventDefault();
      return;
    }

    e.preventDefault();
    e.stopPropagation();

    const touch = e.touches[0];
    const elementRect = (e.target as HTMLElement).getBoundingClientRect();
    const offsetX = touch.clientX - elementRect.left - (elementRect.width / 2);
    const offsetY = touch.clientY - elementRect.top - (elementRect.height / 2);

    setDragOffset({ x: offsetX, y: offsetY });
    setTouchDragItem({ ...accessPoint, isNew: false } as any);
    setIsDraggingTouch(true);

    setTouchDragPreview({
      x: touch.clientX + TOUCH.PREVIEW_OFFSET_X,
      y: touch.clientY + TOUCH.PREVIEW_OFFSET_Y,
      visible: true
    });
  }, [allowNodeMovement]);

  // Handle drop
  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    if (!draggedItem || !mapContainerRef.current || !mapImageRef.current) return;

    const { x, y } = screenToMapCoordinates(
      e.clientX,
      e.clientY,
      mapContainerRef,
      mapImageRef,
      state.viewport.zoomLevel,
      state.viewport.panOffset,
      state.viewport.rotation,
      state.imageDimensions
    );

    const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);

    if (draggedItem.isNew) {
      if (isTypeAtLimit(draggedItem.type, accessPoints)) {
        setDraggedItem(null);
        setDragOffset({ x: 0, y: 0 });
        return;
      }

      const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);

      const newAccessPoint: AccessPointMarkerPoint = {
        id: generateUniqueId(draggedItem.type, accessPoints),
        type: draggedItem.type,
        x: x,
        y: y,
        relativeX,
        relativeY,
        page: currentPage,
        name: `${accessPointTypes.find(t => t.id === draggedItem.type)?.name}`,
        floor: '',
        deviceDetails: generateDeviceDetails(draggedItem.type)
      };
      setAccessPoints(prev => [...prev, newAccessPoint]);
    } else if (allowNodeMovement && draggedItem.id) {
      const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);
      setAccessPoints(prev => prev.map(ap =>
        ap.id === draggedItem.id
          ? {
            ...ap,
            x: x,
            y: y,
            relativeX,
            relativeY,
            page: currentPage
          }
          : ap
      ));
    }

    setDraggedItem(null);
    setDragOffset({ x: 0, y: 0 });
  }, [draggedItem, dragOffset, accessPoints, currentPage, allowNodeMovement, state, mapContainerRef, mapImageRef, accessPointTypes, generateDeviceDetails, setAccessPoints]);

  const handleDragOver = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.dataTransfer.dropEffect = draggedItem?.isNew ? 'copy' : 'move';
  }, [draggedItem]);

  // Handle touch move (mobile)
  const handleTouchMove = useCallback((e: TouchEvent) => {
    if (!isDraggingTouch || !touchDragItem) return;

    e.preventDefault();
    const touch = e.touches[0];
    setTouchDragPreview({
      x: touch.clientX + TOUCH.PREVIEW_OFFSET_X,
      y: touch.clientY + TOUCH.PREVIEW_OFFSET_Y,
      visible: true
    });
  }, [isDraggingTouch, touchDragItem]);

  // Handle touch end (mobile)
  const handleTouchEnd = useCallback((e: TouchEvent) => {
    if (!isDraggingTouch || !touchDragItem) return;

    e.preventDefault();
    const touch = e.changedTouches[0];

    const mapElement = mapContainerRef.current;
    if (mapElement && mapImageRef.current) {
      const rect = mapElement.getBoundingClientRect();
      const isOverMap = touch.clientX >= rect.left &&
        touch.clientX <= rect.right &&
        touch.clientY >= rect.top &&
        touch.clientY <= rect.bottom;

      if (isOverMap) {
        const { x, y } = screenToMapCoordinates(
          touch.clientX - dragOffset.x,
          touch.clientY - dragOffset.y,
          mapContainerRef,
          mapImageRef,
          state.viewport.zoomLevel,
          state.viewport.panOffset,
          state.viewport.rotation,
          state.imageDimensions
        );

        const currentDimensions = getCurrentImageDimensions(mapImageRef, state.imageDimensions);

        if (touchDragItem.isNew) {
          const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);

          const newAccessPoint: AccessPointMarkerPoint = {
            id: generateUniqueId(touchDragItem.type, accessPoints),
            type: touchDragItem.type,
            x: x,
            y: y,
            relativeX,
            relativeY,
            page: currentPage,
            name: `${accessPointTypes.find(t => t.id === touchDragItem.type)?.name}`,
            floor: '',
            deviceDetails: generateDeviceDetails(touchDragItem.type)
          };
          setAccessPoints(prev => [...prev, newAccessPoint]);
        } else if (allowNodeMovement && touchDragItem.id) {
          const { relativeX, relativeY } = pixelToRelative(x, y, currentDimensions.width, currentDimensions.height);

          setAccessPoints(prev => prev.map(ap =>
            ap.id === touchDragItem.id
              ? {
                ...ap,
                x: x,
                y: y,
                relativeX,
                relativeY,
                page: currentPage
              }
              : ap
          ));
        }
      }
    }

    setIsDraggingTouch(false);
    setTouchDragItem(null);
    setTouchDragPreview({ x: 0, y: 0, visible: false });
    setDragOffset({ x: 0, y: 0 });
  }, [isDraggingTouch, touchDragItem, dragOffset, accessPoints, currentPage, allowNodeMovement, state, mapContainerRef, mapImageRef, accessPointTypes, generateDeviceDetails, setAccessPoints]);

  useEffect(() => {
    document.addEventListener('touchmove', handleTouchMove, { passive: false });
    document.addEventListener('touchend', handleTouchEnd, { passive: false });

    return () => {
      document.removeEventListener('touchmove', handleTouchMove);
      document.removeEventListener('touchend', handleTouchEnd);
    };
  }, [handleTouchMove, handleTouchEnd]);

  return {
    draggedItem,
    touchDragItem,
    isDraggingTouch,
    touchDragPreview,
    dragOffset,
    limitMessage,
    handleDragStart,
    handleTouchStartType,
    handleAccessPointDragStart,
    handleTouchStartAccessPoint,
    handleDrop,
    handleDragOver,
    handleTouchMove,
    handleTouchEnd,
    setLimitMessage
  };
};