"use client"
import React from "react";

import AP from "@/app/types/AP";
import { SCREEN_TYPES } from "@/features/project-setup/utils/constants";

// Define the state interface
interface LoaderState {
  loading: boolean;
  currentScreen: string;
  globalAps: AP[];
  currentStep: number;
  completedSteps: number[];
}

// Define action types
type LoaderAction =
  | { type: 'SET_LOADING'; payload: boolean }
  | { type: 'SET_CURRENT_SCREEN'; payload: string }
  | { type: 'SET_GLOBAL_APS'; payload: AP[] }
  | { type: 'SET_CURRENT_STEP'; payload: number }
  | { type: 'COMPLETE_CURRENT_STEP' }
  | { type: 'COMPLETE_STEP'; payload: number }
  | { type: 'SET_COMPLETED_STEPS'; payload: number[] };

// Define the context type
interface LoaderContextType {
  loading: boolean;
  setLoading: (value: boolean) => void;
  currentScreen: string;
  setCurrentScreen: (value: string) => void;
  globalAps: AP[];
  setGlobalAps: (value: AP[]) => void;
  currentStep: number;
  setCurrentStep: (value: number | ((prev: number) => number)) => void;
  completedSteps: number[];
  setCompletedSteps: (value: number[]) => void;
  completeCurrentStep: () => void;
  completeStep: (value: number) => void;
}

// Helper function to get screen from step
function getScreenFromStep(step: number): string {
  switch (step) {
  case 0:
    return SCREEN_TYPES.WELCOME;
  case 1:
    return SCREEN_TYPES.LOADING_NETWORK;
  case 2:
    return SCREEN_TYPES.LOADING_AP;
  case 3:
    return SCREEN_TYPES.UPLOAD;
  case 4:
    return SCREEN_TYPES.PDF_CROP;
  case 5:
    return SCREEN_TYPES.PROJECT_SETUP;
  case 6:
    return SCREEN_TYPES.PROJECT_MODAL;
  default:
    return SCREEN_TYPES.WELCOME;
  }
}

// Reducer function
function loaderReducer(state: LoaderState, action: LoaderAction): LoaderState {
  switch (action.type) {
  case 'SET_LOADING':
    return { ...state, loading: action.payload };

  case 'SET_CURRENT_SCREEN':
    return { ...state, currentScreen: action.payload };

  case 'SET_GLOBAL_APS':
    return { ...state, globalAps: action.payload };

  case 'SET_CURRENT_STEP':
    // Update both currentStep and currentScreen based on step
    return {
      ...state,
      currentStep: action.payload,
      currentScreen: getScreenFromStep(action.payload)
    };

  case 'COMPLETE_CURRENT_STEP':
    // Add current step to completedSteps if not already present
    if (state.completedSteps.includes(state.currentStep)) {
      return state;
    }
    return {
      ...state,
      completedSteps: [...state.completedSteps, state.currentStep].sort((a, b) => a - b)
    };

  case 'COMPLETE_STEP':
    // Add specific step to completedSteps if not already present
    if (state.completedSteps.includes(action.payload)) {
      return state;
    }
    return {
      ...state,
      completedSteps: [...state.completedSteps, action.payload].sort((a, b) => a - b)
    };

  case 'SET_COMPLETED_STEPS':
    return { ...state, completedSteps: action.payload };

  default:
    return state;
  }
}

// Screens that should NOT be persisted (work-in-progress screens)
// These screens represent active editing/work states that should reset to WELCOME
const NON_PERSISTABLE_SCREENS = [
  SCREEN_TYPES.PROJECT_SETUP,  // Main map editing screen - active work in progress
  SCREEN_TYPES.PDF_CROP,       // PDF cropping screen - active editing
  SCREEN_TYPES.UPLOAD,         // Upload screen - transient state
  SCREEN_TYPES.FAIL,           // Error screen - should start fresh
  SCREEN_TYPES.LOADING_NETWORK, // Loading states should not persist
  SCREEN_TYPES.LOADING_AP,
  SCREEN_TYPES.MODULE_LOADING,
];

// Helper function to load state from sessionStorage
function loadPersistedState(): LoaderState {
  if (typeof window === 'undefined') {
    return {
      loading: false,
      currentScreen: SCREEN_TYPES.WELCOME,
      globalAps: [],
      currentStep: 0,
      completedSteps: []
    };
  }

  try {
    const persistedState = sessionStorage.getItem('loaderState');
    if (persistedState) {
      const parsed = JSON.parse(persistedState);
      const persistedScreen = parsed.currentScreen || SCREEN_TYPES.WELCOME;

      // If the persisted screen is a non-persistable screen (work-in-progress or error),
      // reset to WELCOME to avoid bugginess when user returns after tab switch
      const shouldReset = NON_PERSISTABLE_SCREENS.includes(persistedScreen);

      if (shouldReset) {
        // Clear the sessionStorage completely when resetting
        // This ensures a fresh start without any leftover progress
        sessionStorage.removeItem('loaderState');
        console.log('[LoaderContext] Detected non-persistable screen, resetting to WELCOME');
      }

      const currentScreen = shouldReset ? SCREEN_TYPES.WELCOME : persistedScreen;

      // Also reset completedSteps and currentStep if we're resetting the screen
      // This ensures the step progress indicators show correctly
      return {
        loading: false, // Never persist loading state
        currentScreen,
        globalAps: shouldReset ? [] : (parsed.globalAps || []),
        currentStep: shouldReset ? 0 : (parsed.currentStep || 0),
        completedSteps: shouldReset ? [] : (parsed.completedSteps || [])
      };
    }
  } catch (error) {
    console.error('Failed to load persisted loader state:', error);
  }

  return {
    loading: false,
    currentScreen: SCREEN_TYPES.WELCOME,
    globalAps: [],
    currentStep: 0,
    completedSteps: []
  };
}

// Initial state
const initialState: LoaderState = loadPersistedState();

// Create the context with default values
const LoaderContext = React.createContext<LoaderContextType | undefined>(undefined);

// Custom hook to use the LoaderContext
export const useLoader = () => {
  const context = React.useContext(LoaderContext);
  if (!context) {
    throw new Error("useLoader must be used within a LoaderProvider");
  }
  return context;
};

// Provider component
export const LoaderProvider = ({ children }: { children: React.ReactNode }) => {
  const [state, dispatch] = React.useReducer(loaderReducer, initialState);

  // Persist state to sessionStorage whenever it changes (except loading)
  React.useEffect(() => {
    if (typeof window !== 'undefined') {
      try {
        const stateToPersist = {
          currentScreen: state.currentScreen,
          globalAps: state.globalAps,
          currentStep: state.currentStep,
          completedSteps: state.completedSteps
        };
        sessionStorage.setItem('loaderState', JSON.stringify(stateToPersist));
      } catch (error) {
        console.error('Failed to persist loader state:', error);
      }
    }
  }, [state.currentScreen, state.globalAps, state.currentStep, state.completedSteps]);

  // Create stable callback functions that dispatch actions
  const setLoading = React.useCallback((value: boolean) => {
    dispatch({ type: 'SET_LOADING', payload: value });
  }, []);

  const setCurrentScreen = React.useCallback((value: string) => {
    dispatch({ type: 'SET_CURRENT_SCREEN', payload: value });
  }, []);

  const setGlobalAps = React.useCallback((value: AP[]) => {
    dispatch({ type: 'SET_GLOBAL_APS', payload: value });
  }, []);

  const setCurrentStep = React.useCallback((value: number | ((prev: number) => number)) => {
    if (typeof value === 'function') {
      // Handle updater function - compute new value from current state
      const newStep = value(state.currentStep);
      dispatch({ type: 'SET_CURRENT_STEP', payload: newStep });
    } else {
      dispatch({ type: 'SET_CURRENT_STEP', payload: value });
    }
  }, [state.currentStep]);

  const setCompletedSteps = React.useCallback((value: number[]) => {
    dispatch({ type: 'SET_COMPLETED_STEPS', payload: value });
  }, []);

  const completeCurrentStep = React.useCallback(() => {
    dispatch({ type: 'COMPLETE_CURRENT_STEP' });
  }, []);

  const completeStep = React.useCallback((stepIndex: number) => {
    dispatch({ type: 'COMPLETE_STEP', payload: stepIndex });
  }, []);

  // Memoize the context value to prevent unnecessary re-renders
  const contextValue = React.useMemo(() => ({
    loading: state.loading,
    setLoading,
    currentScreen: state.currentScreen,
    setCurrentScreen,
    globalAps: state.globalAps,
    setGlobalAps,
    currentStep: state.currentStep,
    setCurrentStep,
    completedSteps: state.completedSteps,
    setCompletedSteps,
    completeCurrentStep,
    completeStep
  }), [
    state.loading,
    state.currentScreen,
    state.globalAps,
    state.currentStep,
    state.completedSteps,
    setLoading,
    setCurrentScreen,
    setGlobalAps,
    setCurrentStep,
    setCompletedSteps,
    completeCurrentStep,
    completeStep
  ]);

  return (
    <LoaderContext.Provider value={contextValue}>
      {children}
    </LoaderContext.Provider>
  );
};
