"use client"

import React from 'react';
import { SplashScreen } from '@/shared/components';
import branding from '@/lib/branding';

/**
 * Context for managing login splash screen state
 */
interface LoginSplashContextType {
  showLoginSplash: boolean;
  loginSplashFadeOut: boolean;
  setShowLoginSplash: (show: boolean) => void;
  setLoginSplashFadeOut: (fadeOut: boolean) => void;
  hideLoginSplash: () => void;
}

const LoginSplashContext = React.createContext<LoginSplashContextType | undefined>(undefined);

/**
 * LoginSplashProvider Component
 *
 * Wraps the application and shows a splash screen after successful login.
 * Provides methods to control the splash screen from login and home pages.
 */
export function LoginSplashProvider({ children }: { children: React.ReactNode }) {
  const [showLoginSplash, setShowLoginSplash] = React.useState(false);
  const [loginSplashFadeOut, setLoginSplashFadeOut] = React.useState(false);

  const hideLoginSplash = React.useCallback(() => {
    setShowLoginSplash(false);
    setLoginSplashFadeOut(false);
  }, []);

  const value = React.useMemo(
    () => ({
      showLoginSplash,
      loginSplashFadeOut,
      setShowLoginSplash,
      setLoginSplashFadeOut,
      hideLoginSplash,
    }),
    [showLoginSplash, loginSplashFadeOut, hideLoginSplash]
  );

  return (
    <LoginSplashContext.Provider value={value}>
      {children}
      <SplashScreen
        show={showLoginSplash}
        fadeOut={loginSplashFadeOut}
        title={branding.productName}
        subtitle="Welcome back!"
        showSpinner={true}
      />
    </LoginSplashContext.Provider>
  );
}

/**
 * Hook to access login splash context
 */
export function useLoginSplash() {
  const context = React.useContext(LoginSplashContext);
  if (context === undefined) {
    throw new Error('useLoginSplash must be used within a LoginSplashProvider');
  }
  return context;
}
