import { MAX_ZOOM, MIN_ZOOM, ZOOM_STEP } from "@/lib/nodeScaling";

export const initialState = {
  viewport: {
    zoomLevel: 1,
    panOffset: { x: 0, y: 0 },
    rotation: 0,
    isPanning: false,
    lastPanPosition: { x: 0, y: 0 }
  },
  accessPoints: [],
  imageDimensions: { width: 0, height: 0 },
  draggedItem: null,
  isNew: false,
  src: null,
  touch: {
    initialDistance: null,
    initialZoom: 1,
    isPinching: false,
    touchCount: 0
  },
  crop: {
    overflow: 'hidden',
    width: 'none',
    height: 'none'
  },
  pdfFile: null,
  ui: {
    showControls: false,
  }
};
export function viewportReducer(state: { viewport?: any; imageDimensions?: any; accessPoints?: any; touch?: any; }, action: { type: any; payload: number; }) : object {
  switch (action.type) {
  case 'SET_ZOOM':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: Math.max(MIN_ZOOM, Math.min(MAX_ZOOM, action.payload))
      }
    };

  case 'ZOOM_IN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: Math.min(MAX_ZOOM, state.viewport.zoomLevel + ZOOM_STEP)
      }
    };

  case 'ZOOM_OUT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: Math.max(MIN_ZOOM, state.viewport.zoomLevel - ZOOM_STEP)
      }
    };

  case 'SET_PAN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        panOffset: action.payload
      }
    };

  case 'START_PAN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        isPanning: true,
        lastPanPosition: action.payload
      }
    };

  case 'UPDATE_PAN':
    if (!state.viewport.isPanning) return state;

    const deltaX = action.payload.x - state.viewport.lastPanPosition.x;
    const deltaY = action.payload.y - state.viewport.lastPanPosition.y;

    return {
      ...state,
      viewport: {
        ...state.viewport,
        panOffset: {
          x: state.viewport.panOffset.x + deltaX,
          y: state.viewport.panOffset.y + deltaY
        },
        lastPanPosition: action.payload
      }
    };

  case 'END_PAN':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        isPanning: false
      }
    };

  case 'SET_ROTATION':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        rotation: ((action.payload % 360) + 360) % 360
      }
    };

  case 'ROTATE_LEFT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        rotation: ((state.viewport.rotation - 90) + 360) % 360
      }
    };

  case 'ROTATE_RIGHT':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        rotation: (state.viewport.rotation + 90) % 360
      }
    };

  case 'RESET_VIEW':
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: 1,
        panOffset: { x: 0, y: 0 },
        rotation: 0
      }
    };

  case 'FIT_TO_VIEW':
    // Calculate zoom to fit image in container
    const containerPadding = 40;
    const containerWidth = action.payload.containerWidth - containerPadding;
    const containerHeight = action.payload.containerHeight - containerPadding;
    const imageWidth = state.imageDimensions.width;
    const imageHeight = state.imageDimensions.height;

    if (imageWidth === 0 || imageHeight === 0) return state;

    const scaleX = containerWidth / imageWidth;
    const scaleY = containerHeight / imageHeight;
    const fitZoom = Math.min(scaleX, scaleY, MAX_ZOOM);

    // Center the image
    const panX = (containerWidth - imageWidth * fitZoom) / 2;
    const panY = (containerHeight - imageHeight * fitZoom) / 2;

    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: fitZoom,
        panOffset: { x: panX, y: panY },
      }
    };

  case 'ADD_ACCESS_POINT':
    return {
      ...state,
      accessPoints: [...state.accessPoints, action.payload]
    };
  case 'FIX_CROP':
    return {
      ...state,
      crop: {
        overflow: 'hidden',
        width: 'none',
        height: 'none'
      },
    };
  case 'RESET_CROP':
    return {
      ...state,
      crop: {
        overflow: 'hidden',
        width: '100%',
        height: '100vh'
      },
    };
  case 'UPDATE_ACCESS_POINT':
    return {
      ...state,
      accessPoints: state.accessPoints.map(ap =>
        ap.id === action.payload.id ? { ...ap, ...action.payload.updates } : ap
      )
    };
  case 'UPDATE_SRC':
    return {
      ...state,
      src: action.payload
    };
  case 'SET_DRAGGED_ITEM':
    return {
      ...state,
      draggedItem: action.payload
    };
  case 'SET_DRAGGED_ITEM_NEW':
    return {
      ...state,
      draggedItem: action.payload,
      isNew: true
    };

  case 'SET_IMAGE_DIMENSIONS':
    return {
      ...state,
      imageDimensions: action.payload
    };
  case 'SET_PDF_FILE':
    return {
      ...state,
      pdfFile: action.payload
    };
  case 'SET_TOUCH':
    return {
      ...state,
      touch: { ...state.touch, ...action.payload }
    };

  case 'CROP_TO_VIEWPORT':
    // Calculate visible area and create cropped image
    return {
      ...state,
      viewport: {
        ...state.viewport,
        zoomLevel: 1,
        panOffset: { x: 0, y: 0 },
        // Keep rotation as is
      },
      // Access points will be updated separately
      imageDimensions: action.payload.newDimensions
    };

  case 'UPDATE_AFTER_CROP':
    return {
      ...state,
      accessPoints: action.payload.accessPoints,
      imageDimensions: action.payload.imageDimensions
    };

  default:
    return state;
  }
}
