#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <json-c/json.h>

#include "controller.h"
#include "log.h"
#include "http.h"
#include "mac.h"

#define STATSD_PATH	"/tmp/statsd.json"

char ever_read;

static char *uconfigd_controller_subscription_key(const char *json_str)
{
	struct json_object *jobj, *jobj_key;
	char *key = NULL;

	jobj = json_tokener_parse(json_str);
	if (!jobj) {
		MSG(ERROR, "Failed to parse json\n");
		return NULL;
	}

	jobj_key = json_object_object_get(jobj, "key");
	if (!jobj_key) {
		MSG(ERROR, "Failed to get key\n");
		goto out;
	}

	key = strdup(json_object_get_string(jobj_key));

out:
	json_object_put(jobj);

	return key;
}

int uconfigd_controller_subscribe()
{
	FILE *fp;
	char buf[1024] = {0};
	char received[1024] = {0};
	char *key;
	int ret;

	/* Read Subscription object from file */
	fp = fopen(UCONFIGD_SUBSCRIPTION_OBJECT_PATH, "r");
	if (!fp) {
		MSG(ERROR, "Failed to open subscription object\n");
		return -1;
	}

	ret = fread(buf, 1, sizeof(buf), fp);
	if (ret < 0) {
		MSG(ERROR, "Failed to read subscription object\n");
		fclose(fp);
		return -1;
	}
	fclose(fp);


	ret = uconfigd_http_post(UCONFIGD_HTTP_CONTROLLER_BASE_URL "/node/subscribe", buf, received, sizeof(received));
	if (ret < 0)
		return ret;

	/* Check if return code is 200 or 201 */
	if (ret != 200 && ret != 201) {
		MSG(ERROR, "Failed to subscribe: status=%d content=%s\n", ret, received);
		return -1;
	}

	/* Parse json and save key if available */
	key = uconfigd_controller_subscription_key(received);
	if (!key) {
		/* No key is okay */
		return 0;
	}

	fp = fopen(UCONFIGD_SUBSCRIPTION_KEY, "w");
	if (!fp) {
		MSG(ERROR, "Failed to open subscription object\n");
		goto out;
	}

	if (fwrite(key, 1, strlen(key), fp) != strlen(key)) {
		MSG(ERROR, "Failed to write subscription object\n");
		goto out;
	}

out:
	if (fp)
		fclose(fp);
	
	if (key)
		free(key);

	return 0;
}

int uconfigd_controller_reachable()
{
	char received[512] = {0};
	int ret;

	ret = uconfigd_http_get(UCONFIGD_HTTP_CONTROLLER_BASE_URL "/node/test", received, sizeof(received));
	if (ret < 0) {
		MSG(ERROR, "Failed to reach controller\n");
		return 0;
	}

	/* Check if return code is 200 or 201 */
	if (ret != 200) {
		MSG(ERROR, "Failed to subscribe: status=%d content=%s\n", ret, received);
		return 0;
	}

	return 1;
}

int uconfigd_controller_config_retrieve(uint8_t *mac_address, const char *destination_path)
{
	char mac_address_str[UCONFIGD_ETH_STRLEN];
	char config_content[16284];
	char url[256];
	int ret = 0;

	uconfigd_mac_address_string(mac_address, mac_address_str);

	snprintf(url, sizeof(url), UCONFIGD_HTTP_CONTROLLER_BASE_URL "/node/%s/config", mac_address_str);

	/* ToDo: Write directly to output file */
	if (uconfigd_http_get(url, config_content, sizeof(config_content)) != 200) {
		if(ever_read)
			MSG(ERROR, "Failed to retrieve config\n");
		return -1;
	}
	if(ever_read==0){
		MSG(INFO, "Retrieved the config from controller for the first time.\n");
		ever_read = 1;
	}

	FILE *fp = fopen(destination_path, "w");
	if (!fp) {
		MSG(ERROR, "Failed to open destination file\n");
		return -1;
	}

	if (fwrite(config_content, 1, strlen(config_content), fp) != strlen(config_content)) {
		MSG(ERROR, "Failed to write config to file\n");
		fclose(fp);
		return -1;
	}

	fclose(fp);
	return ret;
}

int uconfigd_controller_statistics_submit(uint8_t *mac_address)
{
	char mac_address_str[UCONFIGD_ETH_STRLEN];
	char url[256];
	int ret = 0;

	/* Check if statsd file exists */
	if (access(STATSD_PATH, F_OK) == -1) {
		MSG(INFO, "No statsd file found\n");
		return 0;
	}

	uconfigd_mac_address_string(mac_address, mac_address_str);

	snprintf(url, sizeof(url), UCONFIGD_HTTP_CONTROLLER_BASE_URL "/node/%s/metrics", mac_address_str);

	ret = uconfigd_http_post_file(url, STATSD_PATH);
	if (ret != 200) {
		MSG(VERBOSE, "Failed to submit statsd url=%s response=%d\n", url, ret);
		return -1;
	}

	return ret != 200;
}
