#include <stdio.h>
#include <stdlib.h>
#include <libubus.h>

#include "log.h"

#include "ubus.h"

static struct blob_buf b;

static char*
uconfigd_ubus_get_state_string(enum uconfig_ubus_apply_state state)
{
	switch (state) {
	case UCONFIG_UBUS_APPLY_STATE_IDLE:
		return "idle";
	case UCONFIG_UBUS_APPLY_STATE_PENDING_DOWNLOAD:
		return "pending-download";
	case UCONFIG_UBUS_APPLY_STATE_PENDING_APPLY:
		return "pending-apply";
	case UCONFIG_UBUS_APPLY_STATE_APPLIED:
		return "applied";
	case UCONFIG_UBUS_APPLY_STATE_FAILED:
		return "failed";
	case UCONFIG_UBUS_APPLY_STATE_ERROR:
		return "error";
	case UCONFIG_UBUS_APPLY_STATE_ADOPT:
		return "adopt";
	default:
		return "unknown";
	}
}

static int
uconfigd_ubus_info(struct ubus_context *ctx, struct ubus_object *obj,
		   struct ubus_request_data *req, const char *method,
		   struct blob_attr *msg)
{
	struct uconfigd_ubus *uub = container_of(ctx, struct uconfigd_ubus, ctx);
	void *config_table, *config_revision_table;

	blob_buf_init(&b, 0);

	blobmsg_add_string(&b, "version", "0.1");
	config_table = blobmsg_open_table(&b, "config");

	/* Current state */
	blobmsg_add_string(&b, "state", uconfigd_ubus_get_state_string(uub->apply.state));

	/* Current config version */
	config_revision_table = blobmsg_open_table(&b, "current");
	blobmsg_close_table(&b, config_revision_table);

	/* Pending config version */
	config_revision_table = blobmsg_open_table(&b, "pending");
	blobmsg_close_table(&b, config_revision_table);

	blobmsg_close_table(&b, config_table);

	ubus_send_reply(ctx, req, b.head);

	return UBUS_STATUS_OK;
}

enum uconfigd_ubus_apply_policy_names {
	UCONFIGD_APPLY_PATH = 0,
	__UCONFIGD_APPLY_MAX
};

static const struct blobmsg_policy uconfigd_ubus_apply_policy[__UCONFIGD_APPLY_MAX] = {
	[UCONFIGD_APPLY_PATH] = { .name = "path", .type = BLOBMSG_TYPE_STRING },
};

static int
uconfigd_ubus_apply(struct ubus_context *ctx, struct ubus_object *obj,
		    struct ubus_request_data *req, const char *method,
		    struct blob_attr *msg)
{
	struct uconfigd_ubus *uub = container_of(ctx, struct uconfigd_ubus, ctx);
	struct blob_attr *tb[__UCONFIGD_APPLY_MAX];
	char cmd_buf[512] = {0};
	FILE *fp;
	int ret;
	MSG(INFO, "Save config\n");

	/* Parse blobmsg */
	ret = blobmsg_parse(uconfigd_ubus_apply_policy, __UCONFIGD_APPLY_MAX, tb, blob_data(msg), blob_len(msg));
	if (ret) {
		MSG(ERROR, "Failed to parse blobmsg\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	/* Check if path is set */
	if (!tb[UCONFIGD_APPLY_PATH]) {
		MSG(ERROR, "Path not set\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	ret = snprintf(cmd_buf, sizeof(cmd_buf), "render_config -t %s", blobmsg_get_string(tb[UCONFIGD_APPLY_PATH]));
	if (ret < 0 || ret >= sizeof(cmd_buf)) {
		MSG(ERROR, "Failed to create command\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	/* Validate provided file */
	/* Call render_config -t <filename> */
	fp = popen(cmd_buf, "r");
	ret = pclose(fp);
	if (ret) {
		MSG(ERROR, "Failed to render config\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	/* Link to /etc/uconfig/uniberg/pending.json */
	ret = snprintf(cmd_buf, sizeof(cmd_buf), "cp %s /etc/uconfig/uconfig.pending",
		       blobmsg_get_string(tb[UCONFIGD_APPLY_PATH]));
	if (ret < 0 || ret >= sizeof(cmd_buf)) {
		MSG(ERROR, "Failed to create command\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}
	ret = system(cmd_buf);
	if (ret) {
		MSG(ERROR, "Failed to link config\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	/* Schedule config apply */
	MSG(INFO, "Schedule config apply\n");
	uub->apply.state = UCONFIG_UBUS_APPLY_STATE_PENDING_DOWNLOAD;

	return UBUS_STATUS_OK;
}

static int uconfigd_ubus_confirm_apply(struct ubus_context *ctx, struct ubus_object *obj,
				       struct ubus_request_data *req, const char *method,
				       struct blob_attr *msg)
{
	struct uconfigd_ubus *uub = container_of(ctx, struct uconfigd_ubus, ctx);
	FILE *fp;
	int ret;

	MSG(INFO, "Confirm apply\n");

	if (uub->apply.state != UCONFIG_UBUS_APPLY_STATE_APPLIED) {
		MSG(ERROR, "Invalid state\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	/* Make the pending config active */
	fp = popen("mv /etc/uconfig/uconfig.pending /etc/uconfig/uconfig.active", "r");
	ret = pclose(fp);
	if (ret) {
		MSG(ERROR, "Failed to save apply state\n");
		return UBUS_STATUS_INVALID_ARGUMENT;
	}

	uub->apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
	return UBUS_STATUS_OK;
}

static const struct ubus_method uconfigd_ubus_methods[] = {
	UBUS_METHOD_NOARG("info", uconfigd_ubus_info),
	UBUS_METHOD("apply", uconfigd_ubus_apply, uconfigd_ubus_apply_policy),
	UBUS_METHOD_NOARG("apply_confirm", uconfigd_ubus_confirm_apply),
};

static struct ubus_object_type uconfigd_ubus_obj_type =
	UBUS_OBJECT_TYPE("uconfigd", uconfigd_ubus_methods);

struct ubus_object uconfigd_ubus_obj = {
	.name = "uconfigd",
	.type = &uconfigd_ubus_obj_type,
	.methods = uconfigd_ubus_methods,
	.n_methods = ARRAY_SIZE(uconfigd_ubus_methods),
};

int uconfigd_ubus_start(struct uconfigd_ubus *uub)
{
	int ret;
	MSG(INFO, "Connecting to ubus\n");

	ret = ubus_connect_ctx(&uub->ctx, NULL);
	if (ret) {
		MSG(ERROR, "Failed to connect to ubus: %s\n", ubus_strerror(ret));
		return -1;
	}

	ret = ubus_add_object(&uub->ctx, &uconfigd_ubus_obj);
	if (ret) {
		MSG(ERROR, "Failed to add ubus object: %s\n", ubus_strerror(ret));
		ubus_free(&uub->ctx);
		return -1;
	}

	ubus_add_uloop(&uub->ctx);

	return 0;
}

int uconfigd_ubus_stop(struct uconfigd_ubus *uub)
{
	MSG(INFO, "Disconnecting from ubus\n");
	return 0;
}
