#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/sysinfo.h>
#include <json-c/json.h>

#include "announcement.h"
#include "controller.h"
#include "log.h"
#include "ubus.h"
#include "uconfigd.h"

#define UCONFIGD_LABEL_MAC_PATH "/lib/uniberg/uconfig/system/label_mac"


static int uconfigd_core_update_system_uptime(struct uconfigd_data *ucd)
{
	struct sysinfo si = {};

	if (sysinfo(&si)) {
		MSG(ERROR, "Failed to get system uptime\n");
		return -1;
	}

	ucd->system.uptime_current = (uint64_t) si.uptime;

	return 0;
}

static int uconfigd_label_mac_read(struct uconfigd_data *ucd)
{
	/* Read label-mac from file */
	int matched;
	FILE *fp;

	fp = fopen(UCONFIGD_LABEL_MAC_PATH, "r");
	if (!fp) {
		MSG(ERROR, "Failed to open label-mac\n");
		return -1;
	}

	matched = fscanf(fp, "%hhx:%hhx:%hhx:%hhx:%hhx:%hhx",
			&ucd->label_mac[0], &ucd->label_mac[1], &ucd->label_mac[2],
			&ucd->label_mac[3], &ucd->label_mac[4], &ucd->label_mac[5]);
	if (matched != 6) {
		MSG(ERROR, "Failed to read label-mac file=%s\n", UCONFIGD_LABEL_MAC_PATH);
		fclose(fp);
		return -1;
	}

	fclose(fp);
	return 0;
}

static int64_t uconfigd_config_get_uuid(const char *path)
{
	FILE *fp = NULL;
	int64_t ret = -1;
	size_t file_size;
	struct json_object *root_object = NULL;
	struct json_object *uuid_object = NULL;
	char *file_content = NULL;

	fp = fopen(path, "r");
	if (!fp) {
		MSG(ERROR, "Failed to open config file=%s\n", path);
		return -1;
	}

	/* Get File Size */
	fseek(fp, 0, SEEK_END);
	file_size = ftell(fp);

	/* Read File */
	file_content = malloc(file_size);
	if (!file_content) {
		MSG(ERROR, "Failed to allocate memory\n");
		ret = -1;
		goto out;
	}

	fseek(fp, 0, SEEK_SET);
	if (fread(file_content, 1, file_size, fp) != file_size) {
		MSG(ERROR, "Failed to read file\n");
		ret = -1;
		goto out;
	}

	/* Parse JSON */
	root_object = json_tokener_parse(file_content);
	if (!root_object) {
		MSG(ERROR, "Failed to parse JSON\n");
		ret = -1;
		goto out;
	}

	/* Get UUID */
	uuid_object = json_object_object_get(root_object, "uuid");
	if (!uuid_object) {
		MSG(ERROR, "Failed to get UUID\n");
		ret = -1;
		goto out;
	}

	ret = json_object_get_int64(uuid_object);

out:
	if (root_object) {
		json_object_put(root_object);
	}

	if (file_content) {
		free(file_content);
	}

	if (fp) {
		fclose(fp);
	}

	return ret;
}

static int uconfigd_shall_apply(const char *current_path, const char *new_path)
{
	int64_t current_uuid;
	int64_t new_uuid;

	current_uuid = uconfigd_config_get_uuid(current_path);
	new_uuid = uconfigd_config_get_uuid(new_path);

	if (current_uuid < 0) {
		MSG(VERBOSE, "Failed to get current UUID\n");
		/* Not a Fatal, we might have one in new config */
	}

	if (new_uuid < 0) {
		MSG(ERROR, "Failed to get new UUID\n");
		return -1;
	}

	/* Apply if new differs from old and new is valid */
	return current_uuid != new_uuid;
}

static void uconfigd_state_machine_timeout_work(struct uloop_timeout *timeout)
{
	struct uconfigd_data *ucd = container_of(timeout, struct uconfigd_data, state_machine.timeout);
	FILE *fp;
	int ret;

	/* Update current uptime */
	uconfigd_core_update_system_uptime(ucd);

	/* Check if we have a pending apply */
	if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_PENDING_DOWNLOAD) {
		/* Download Config from controller */
		ret = uconfigd_controller_config_retrieve(ucd->label_mac, "/etc/uconfig/uconfig.pending");
		if (ret) {
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
			ucd->ubus.apply.last_changed = ucd->system.uptime_current;
			return;
		} 

		/* Check if UUID changed */
		if (uconfigd_shall_apply("/etc/uconfig/uconfig.active", "/etc/uconfig/uconfig.pending") <= 0) {
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
			ucd->ubus.apply.last_changed = ucd->system.uptime_current;
		} else {
			MSG(VERBOSE, "Pending config apply\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_PENDING_APPLY;
			ucd->ubus.apply.last_changed = ucd->system.uptime_current;
		}
	} else if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_PENDING_APPLY) {
		MSG(WARN, "Applying config\n");

		/* Check if config is valid */
		ret = system("render_config -t /etc/uconfig/uconfig.pending > /tmp/uconfigd.log 2>&1");
		if (ret) {
			MSG(ERROR, "Failed to validate config\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
			ucd->ubus.apply.last_changed = ucd->system.uptime_current;
			return;
		}

		/* Apply Downloaded Config */
		ret = system("render_config /etc/uconfig/uconfig.pending > /tmp/uconfigd.log 2>&1");
		if (ret) {
			MSG(ERROR, "Failed to apply config\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_FAILED;
			return;
		} else {
			MSG(INFO, "Config applied\n");
		}

		ucd->config_uuid = ucd->announcement.config_uuid;

		/* Update State */
		ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_APPLIED;
		ucd->ubus.apply.last_changed = ucd->system.uptime_current;
	} else if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_FAILED) {
		/* Rollback to active config */
		ret = system("render_config /etc/uconfig/uconfig.active > /tmp/uconfigd.log 2>&1");
		if (ret) {
			MSG(ERROR, "Failed to apply config\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_ERROR;
		} else {
			MSG(INFO, "Config applied\n");
		}
		ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
	} else if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_APPLIED) {
		/* Check if we ran into timeout */
		if (ucd->system.uptime_current - ucd->ubus.apply.last_changed > 60) {
			MSG(WARN, "Config apply timeout\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_FAILED;
		}

		/* Check if controller is reachable */
		if (uconfigd_controller_reachable()) {
			MSG(INFO, "Controller reachable\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
			/* Make the pending config active */
			fp = popen("mv /etc/uconfig/uconfig.pending /etc/uconfig/uconfig.active", "r");
			ret = pclose(fp);
			if (ret) {
				MSG(ERROR, "Failed to save apply state\n");
				ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_ERROR;
			}
		} else {
			MSG(ERROR, "Controller not reachable\n");
			/* Do nothing. Timeout at beginning of block will kick in. */
		}
	} else if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_ERROR) {
		MSG(ERROR, "Config apply failed catastrophically\n");
	} else if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_ADOPT) {
		/* Generate Subscription object */
		system("/lib/uniberg/uconfigd/scripts/subscription-object.uc");

		/* Subscribe */
		ret = uconfigd_controller_subscribe();
		if (ret) {
			/* Stay in adoption state */
			MSG(ERROR, "Failed to subscribe\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_ADOPT;
		} else {
			/* Move to idle state */
			MSG(INFO, "Subscribed\n");
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_IDLE;
		}
	} else if (ucd->ubus.apply.state == UCONFIG_UBUS_APPLY_STATE_IDLE) {
		if (0 && ucd->announcement.config_uuid != 0 && ucd->config_uuid != ucd->announcement.config_uuid) {
			/* Received updated config announcement */
			MSG(INFO, "New config available old=%llu\n new=%llu\n", ucd->config_uuid, ucd->announcement.config_uuid);
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_PENDING_DOWNLOAD;
		}

		/* Pull config every X seconds from controller */
		if (ucd->system.uptime_current - ucd->ubus.apply.last_changed > UCONFIGD_CONFIG_DOWNLOAD_INTERVAL) {
			ucd->ubus.apply.last_changed = ucd->system.uptime_current;
			ucd->ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_PENDING_DOWNLOAD;
		}

		/* Submit statistics */
		if (ucd->system.uptime_current - ucd->stats.last_submission > 60) {
			/* Submit stats */
			ucd->stats.last_submission = ucd->system.uptime_current;
			uconfigd_controller_statistics_submit(ucd->label_mac);
		}
	} else {
		MSG(ERROR, "Invalid state\n");
	}
}
static void uconfigd_state_machine_timeout(struct uloop_timeout *timeout)
{
	struct uconfigd_data *ucd = container_of(timeout, struct uconfigd_data, state_machine.timeout);

	/* Do Work */
	uconfigd_state_machine_timeout_work(timeout);
	
	/* Schedule next SM invocation */
	uloop_timeout_set(&ucd->state_machine.timeout, UCONFIGD_STATE_MACHINE_TIMEOUT);

	return;
}

int main(int argc, char *argv[])
{
	struct uconfigd_data ucd = {};
	int ret;

	/* Read label-mac */
	ret = uconfigd_label_mac_read(&ucd);
	if (ret) {
		MSG(ERROR, "Failed to read label-mac\n");
		return -1;
	}

	if (argc > 1) {
		log_level_set(atoi(argv[1]));
	}

	/* Init uptime */
	uconfigd_core_update_system_uptime(&ucd);
	ucd.system.uptime_start = ucd.system.uptime_current;

	/* Init uloop */
	uloop_init();

	/* Init state-machine*/
	ucd.ubus.apply.state = UCONFIG_UBUS_APPLY_STATE_ADOPT;
	ucd.state_machine.timeout.cb = uconfigd_state_machine_timeout;
	uloop_timeout_set(&ucd.state_machine.timeout, UCONFIGD_STATE_MACHINE_TIMEOUT);

	/* Attach to ubus */
	uconfigd_ubus_start(&ucd.ubus);

	/* Receive announcements */
	if (0)
		uconfigd_announcement_start(&ucd.announcement);

	uloop_run();
	uloop_done();

	return 0;
}
