#include <stdlib.h>
#include <string.h>

#include <json-c/json.h>

#include "http.h"
#include "log.h"
#include "permission.h"
#include "sha256.h"

static char *auther_permission_request_data_to_json(const struct auther_permission_request_data *request, char *buf, size_t buflen)
{
	struct json_object *jobj = json_object_new_object();
	char psk_hash[AUTHER_SHA256_DIGEST_BUFFER_LENGTH] = {0};

	if (!jobj) {
		MSG(ERROR, "Failed to create JSON object\n");
		return NULL;
	}

	sha256_string(request->psk, psk_hash);

	json_object_object_add(jobj, "mac_address", json_object_new_string(request->mac));
	json_object_object_add(jobj, "password_hash_type", json_object_new_string("sha256"));
	json_object_object_add(jobj, "password_hash", json_object_new_string(psk_hash));
	json_object_object_add(jobj, "ssid", json_object_new_string(request->ssid));

	const char *json_str = json_object_to_json_string_ext(jobj, JSON_C_TO_STRING_PLAIN);
	if (strlen(json_str) >= buflen) {
		MSG(ERROR, "Buffer too small for JSON string\n");
		json_object_put(jobj);
		return NULL;
	}

	strncpy(buf, json_str, buflen - 1);
	buf[buflen - 1] = '\0';

	json_object_put(jobj);

	return buf;
}

static int auther_permission_request_cache_get(struct auther_permission_data *apd, const struct auther_permission_request_data *request)
{
	struct auther_permission_cache_entry *entry = NULL;
	time_t now = time(NULL);
	int i;

	for (i = 0; i < AUTHER_PERMISSION_CACHE_SIZE; i++) {
		entry = &apd->entries[i];
		if (strcmp(entry->mac, request->mac))
			continue;
		
		if (strcmp(entry->ssid, request->ssid))
			continue;
		
		if (strcmp(entry->psk, request->psk))
			continue;
		
		if (now - entry->timestamp > AUTHER_PERMISSION_CACHE_TIMEOUT)
			continue;
		
		MSG(INFO, "Found cache entry mac=%s ssid=%s psk=%s allowed=%d\n", entry->mac, entry->ssid, entry->psk, entry->allowed);
		return entry->allowed;
	}

	return -1;
}

static int auther_permission_request_cache_put(struct auther_permission_data *apd, const struct auther_permission_request_data *request, int allowed)
{
	struct auther_permission_cache_entry *oldest_entry = NULL;
	struct auther_permission_cache_entry *entry = NULL;
	time_t now = time(NULL);
	int i;

	for (i = 0; i < AUTHER_PERMISSION_CACHE_SIZE; i++) {
		if (!strcmp(apd->entries[i].mac, request->mac)) {
			entry = &apd->entries[i];
			break;
		}

		if (!oldest_entry || oldest_entry->timestamp > apd->entries[i].timestamp) {
			oldest_entry = &apd->entries[i];
			continue;
		}
	}

	if (!entry) {
		if (oldest_entry) {
			entry = oldest_entry;
		} else {
			MSG(ERROR, "Failed to find cache entry\n");
			return -1;
		}
	}

	strncpy(entry->mac, request->mac, sizeof(entry->mac) - 1);
	strncpy(entry->ssid, request->ssid, sizeof(entry->ssid) - 1);
	strncpy(entry->psk, request->psk, sizeof(entry->psk) - 1);
	entry->allowed = allowed;
	entry->timestamp = now;

	MSG(INFO, "Added cache entry mac=%s ssid=%s psk=%s allowed=%d\n", entry->mac, entry->ssid, entry->psk, entry->allowed);

	return -1;
}

int auther_permission_request(struct auther_permission_data *apd, const struct auther_permission_request_data *request)
{
	char buf[1024] = {0};
	char received[1024] = {0};
	int ret;

	ret = auther_permission_request_cache_get(apd, request);
	if (ret >= 0) {
		MSG(INFO, "Found in cache allow=%d\n", ret);

		/* 0 here means "ALLOWED = 1", caller expects 0 = allow 1 = disallow */
		return !ret;
	}

	if (!auther_permission_request_data_to_json(request, buf, sizeof(buf))) {
		MSG(ERROR, "Failed to create JSON object\n");
		return -1;
	}

	ret = auther_http_post("http://[fd42:dead:c0de:1337::1]:8844/wireless/client/bind", buf, received, sizeof(received));
	if (ret < 0) {
		MSG(ERROR, "Failed to request permission status=%d\n", ret);
		return -1;
	}

	/* Check if return code is 200 or 201 */
	if (ret != 200 && ret != 201) {
		MSG(ERROR, "Failed to request permission: status=%d content=%s\n", ret, received);
		ret = -1;
	} else {
		ret = 0;
	}

	auther_permission_request_cache_put(apd, request, ret == 0);

	return ret;
}