#include <stdlib.h>
#include <string.h>

#include <libubus.h>
 
#include "log.h"
#include "permission.h"
#include "ubus.h"

struct blob_buf b;

static struct auther_ubus_hapd *
auther_ubus_hapd_get(struct auther_ubus_ctx *auther_ctx, const char *name, int id)
{
	struct auther_ubus_hapd *hapd;

	/* Check if object exists */
	list_for_each_entry(hapd, &auther_ctx->hapd_list, list) {
		if (hapd->ubus.id == id) {
			return hapd;
		}
	}

	/* Create new object */
	hapd = calloc(sizeof(*hapd), 1);
	if (!hapd) {
		return NULL;
	}

	memcpy(hapd->ubus.name, name, strnlen(name, sizeof(hapd->ubus.name)));
	hapd->ubus.id = id;
	list_add(&hapd->list, &auther_ctx->hapd_list);

	MSG(DEBUG, "Created object %s\n", name);

	return hapd;
}

static void auther_ubus_hapd_free(struct auther_ubus_hapd *hapd)
{
	list_del(&hapd->list);
	free(hapd);
}

static int
auther_hapd_event(struct ubus_context *ctx, struct ubus_object *obj, struct ubus_request_data *req,
		  const char *method, struct blob_attr *msg)
{
	enum {
		PSK_ALLOWED_IFACE,
		PSK_ALLOWED_SSID,
		PSK_ALLOWED_STA,
		PSK_ALLOWED_PSK,
		__PSK_ALLOWED_IFACE_MAX,
	};
	static const struct blobmsg_policy psk_allowed_policy[] = {
		[PSK_ALLOWED_IFACE] = { "iface", BLOBMSG_TYPE_STRING },
		[PSK_ALLOWED_SSID] = { "ssid", BLOBMSG_TYPE_STRING },
		[PSK_ALLOWED_STA] = { "sta", BLOBMSG_TYPE_STRING },
		[PSK_ALLOWED_PSK] = { "psk", BLOBMSG_TYPE_STRING },
	};

	struct auther_ubus_ctx *auther_ctx = container_of(ctx, struct auther_ubus_ctx, ctx);
	struct blob_attr *tb[__PSK_ALLOWED_IFACE_MAX];
	struct auther_permission_request_data auther_request = {}; 
	int allow_connection = 1;
	int ret;

	MSG(DEBUG, "Received event %s\n", method);

	if (strcmp(method, "psk_allowed")) {
		return 0;
	}

	/* Parse Request */
	blobmsg_parse(psk_allowed_policy, __PSK_ALLOWED_IFACE_MAX, tb, blob_data(msg), blob_len(msg));
	if (!tb[PSK_ALLOWED_IFACE] || !tb[PSK_ALLOWED_SSID] || !tb[PSK_ALLOWED_STA] || !tb[PSK_ALLOWED_PSK]) {
		MSG(ERROR, "Invalid request from hostapd iface=%d ssid=%d sta=%d psk=%d\n", !!tb[PSK_ALLOWED_IFACE], !!tb[PSK_ALLOWED_SSID], !!tb[PSK_ALLOWED_STA], !!tb[PSK_ALLOWED_PSK]);
		goto send_reply;
	}

	/* ToDo: Request external source */
	auther_request.ssid = blobmsg_get_string(tb[PSK_ALLOWED_SSID]);
	auther_request.mac = blobmsg_get_string(tb[PSK_ALLOWED_STA]);
	auther_request.psk = blobmsg_get_string(tb[PSK_ALLOWED_PSK]);

	ret = auther_permission_request(&auther_ctx->permission_data, &auther_request);
	if (ret != 0) {
		allow_connection = 0;
	}

	MSG(INFO, "Authentication request iface=%s ssid=%s sta=%s psk=%s allow=%d\n",
	    blobmsg_get_string(tb[PSK_ALLOWED_IFACE]), blobmsg_get_string(tb[PSK_ALLOWED_SSID]), blobmsg_get_string(tb[PSK_ALLOWED_STA]), blobmsg_get_string(tb[PSK_ALLOWED_PSK]), allow_connection);

send_reply:
	/* Send reply */
	blob_buf_init(&b, 0);
	blobmsg_add_u64(&b, "allowed", allow_connection);
	ubus_send_reply(ctx, req, b.head);

	return 0;
}

static void
auther_hapd_event_remove(struct ubus_context *ctx, struct ubus_subscriber *s, unsigned int id)
{
	struct auther_ubus_hapd *hapd = container_of(s, struct auther_ubus_hapd, ubus.subscriber);

	MSG(DEBUG, "Remove hostapd instance id=%d\n", id);

	ubus_unregister_subscriber(ctx, s);
	auther_ubus_hapd_free(hapd);
}

static void auther_ubus_lookup_cb(struct ubus_context *ctx, struct ubus_object_data *obj, void *priv)
{
	struct auther_ubus_ctx *auther_ctx = priv;
	static struct auther_ubus_hapd *hapd;

	MSG(DEBUG, "Found object %s\n", obj->path);

	/* Get Object */
	hapd = auther_ubus_hapd_get(auther_ctx, obj->path, obj->id);
	if (!hapd) {
		MSG(ERROR, "Failed to get object\n");
		return;
	}

	/* Check if already subscribed */
	if (hapd->ubus.subscriber.cb) {
		MSG(DEBUG, "Already subscribed\n");
		return;
	}

	/* Subscribe to object */
	hapd->ubus.subscriber.cb = auther_hapd_event;
	hapd->ubus.subscriber.remove_cb = auther_hapd_event_remove;

	ubus_register_subscriber(&auther_ctx->ctx, &hapd->ubus.subscriber);
	ubus_subscribe(&auther_ctx->ctx, &hapd->ubus.subscriber, obj->id);

	MSG(DEBUG, "Subscribed to object %s\n", obj->path);
}

static int auther_ubus_lookup(struct auther_ubus_ctx *auther_ctx)
{
	return ubus_lookup(&auther_ctx->ctx, "hostapd-auth", auther_ubus_lookup_cb, auther_ctx);
}

static void auther_ubus_event_handler(struct ubus_context *ctx, struct ubus_event_handler *ev, const char *type, struct blob_attr *msg)
{
	struct auther_ubus_ctx *auther_ctx = container_of(ev, struct auther_ubus_ctx, ubus_evh);

	/* Schedule update on hapd instances available */
	auther_ctx->pending_update = 1;
}

static void
auther_ubus_register_events(struct auther_ubus_ctx *auther_ctx)
{
	auther_ctx->ubus_evh.cb = auther_ubus_event_handler;

	/* Update is now pending */
	auther_ctx->pending_update = 1;

	ubus_register_event_handler(&auther_ctx->ctx, &auther_ctx->ubus_evh, "ubus.object.add");
}

int
auther_ubus_recurring_work(struct auther_ubus_ctx *auther_ctx)
{
	if (auther_ctx->pending_update) {
		MSG(DEBUG, "Start ubus object update\n");
		auther_ubus_lookup(auther_ctx);
		auther_ctx->pending_update = 0;
	}

	return 0;
}

int
auther_ubus_start(struct auther_ubus_ctx *auther_ctx)
{
	blob_buf_init(&b, 0);

	INIT_LIST_HEAD(&auther_ctx->hapd_list);

	if (ubus_connect_ctx(&auther_ctx->ctx, NULL)){
		MSG(ERROR, "Failed to connect to ubus\n");
		return -1;
	}

	ubus_add_uloop(&auther_ctx->ctx);

	/* Register for created objects */
	auther_ubus_register_events(auther_ctx);

	return 0;
}
