#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "collector.h"
#include "log.h"
#include "ubus.h"
#include "statsd.h"

#define BATMAN_IFNAME "batman"

static void statsd_write_file(const char *content)
{
	FILE *fp;
	fp = fopen("/tmp/statsd.json", "w");
	if (!fp) {
		MSG_ERROR("Failed to open file\n");
		return;
	}

	fprintf(fp, "%s", content);
	fclose(fp);
}

static void statsd_state_machine_timeout(struct uloop_timeout *timeout)
{
	struct statsd_data *sdd = container_of(timeout, struct statsd_data, state_machine.timeout);
	struct json_object *root_jobj = NULL;
	struct json_object *hostapd_object = NULL;
	struct json_object *jobj = NULL;
	const char *json_string;
	int ret;

	root_jobj = json_object_new_object();
	if (!root_jobj) {
		MSG_ERROR("Failed to create JSON object\n");
		return;
	}

	/* Update Hostapd Data */
	statsd_ubus_hostapd_metrics(&sdd->ubus, &hostapd_object);
	if (hostapd_object)
		json_object_object_add(root_jobj, "clients_hostapd", hostapd_object);

	/* Collect data */
	ret = collector_batman_mesh(&jobj, BATMAN_IFNAME);
	if (ret) {
		MSG_INFO("Failed to collect batman-adv mesh\n");
	} else {
		json_object_object_add(root_jobj, "mesh", jobj);
	}

	ret = collector_batman_neighbors(&jobj, BATMAN_IFNAME);
	if (ret) {
		MSG_INFO("Failed to collect batman-adv neighbors\n");
	} else {
		json_object_object_add(root_jobj, "neighbors", jobj);
	}

	ret = collector_batman_originators(&jobj, BATMAN_IFNAME);
	if (ret) {
		MSG_INFO("Failed to collect batman-adv originators\n");
	} else {
		json_object_object_add(root_jobj, "originators", jobj);
	}

	ret = collector_batman_interfaces(&jobj, BATMAN_IFNAME);
	if (ret) {
		MSG_INFO("Failed to collect batman-adv interfaces\n");
	} else {
		json_object_object_add(root_jobj, "interfaces", jobj);
	}

	json_string = json_object_to_json_string(root_jobj);
	statsd_write_file(json_string);

	/* Print JSON object */
	MSG_INFO("%s\n", json_string);

	json_object_put(root_jobj);
	
	/* Schedule next SM invocation */
	uloop_timeout_set(&sdd->state_machine.timeout, STATSD_STATE_MACHINE_TIMEOUT);

	/* Update hostapd instances */
	statsd_ubus_update_hostapd_instances(&sdd->ubus);

	return;
}

int main(int argc, char *argv[])
{
	struct statsd_data sdd = {};
	int log_level = MSG_ERROR;

	if (argc > 1) {
		log_level = atoi(argv[1]);
	}

	statsd_log_level_set(log_level);

	/* Init uloop */
	uloop_init();

	/* Init state-machine*/
	sdd.state_machine.timeout.cb = statsd_state_machine_timeout;
	uloop_timeout_set(&sdd.state_machine.timeout, STATSD_STATE_MACHINE_TIMEOUT);

	/* Attach to ubus */
	statsd_ubus_start(&sdd.ubus);

	uloop_run();
	uloop_done();

	return 0;
}