#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libubus.h>

#include <libubox/blobmsg_json.h>

#include "log.h"

#include "ubus.h"



static const struct ubus_method statsd_ubus_methods[] = {
};

static struct ubus_object_type statsd_ubus_obj_type =
	UBUS_OBJECT_TYPE("uniberg-statsd", statsd_ubus_methods);

struct ubus_object statsd_ubus_obj = {
	.name = "uniberg-statsd",
	.type = &statsd_ubus_obj_type,
	.methods = statsd_ubus_methods,
	.n_methods = ARRAY_SIZE(statsd_ubus_methods),
};

struct statsd_ubus_hostapd_stat_req {
	struct json_object *jobj;
	struct statsd_ubus_hapd_instance *instance;
};

static void
statsd_ubus_hostapd_metrics_interface_cb(struct ubus_request *req, int type, struct blob_attr *msg)
{
	struct statsd_ubus_hostapd_stat_req *stat_req = req->priv;
	struct json_object *jobj;
	char *str;

	str = blobmsg_format_json_indent(msg, true, 0);

	jobj = json_tokener_parse(str);
	if (!jobj) {
		MSG_ERROR("Failed to parse JSON\n");
		goto out;
	}

	json_object_array_add(stat_req->jobj, jobj);
out:
	free(str);
}

static void statsd_ubus_hostapd_metrics_interface(struct statsd_ubus *uub, struct json_object *jobj, struct statsd_ubus_hapd_instance *instance)
{
	struct statsd_ubus_hostapd_stat_req stat_req = {
		.jobj = jobj,
		.instance = instance,
	};
	MSG_INFO("Requesting hostapd metrics for %s\n", instance->name);

	ubus_invoke(&uub->ctx, instance->id, "get_clients", NULL, statsd_ubus_hostapd_metrics_interface_cb, &stat_req, 500);
}

int statsd_ubus_hostapd_metrics(struct statsd_ubus *uub, struct json_object **jobj)
{
	size_t i;

	*jobj = json_object_new_array();
	if (!*jobj) {
		MSG_ERROR("Failed to create JSON object\n");
		return -1;
	}

	for (i = 0; i < uub->hostapd_instances.count; i++) {
		statsd_ubus_hostapd_metrics_interface(uub, *jobj, &uub->hostapd_instances.instances[i]);
	}

	return 0;
}

static void
statsd_hapd_instances_list_cb(struct ubus_context *ctx, struct ubus_object_data *obj, void *priv)
{
	struct statsd_ubus *uub = container_of(ctx, struct statsd_ubus, ctx);
	struct statsd_ubus_hapd_instance *instance;

	MSG_INFO("Received Instance %s\n", obj->path);

	if (uub->hostapd_instances.count >= 16)
		return;

	instance = &uub->hostapd_instances.instances[uub->hostapd_instances.count++];
	strncpy(instance->name, obj->path, strlen(obj->path));
	instance->id = obj->id;
}

int statsd_ubus_update_hostapd_instances(struct statsd_ubus *uub)
{
	/* Clear existing instances */
	memset(uub->hostapd_instances.instances, 0, sizeof(uub->hostapd_instances.instances));
	uub->hostapd_instances.count = 0;

	MSG_INFO("Lookup Instances\n");

	/* Lookup instances */
	ubus_lookup(&uub->ctx, "hostapd.*", statsd_hapd_instances_list_cb, NULL);
	return 0;
}

int statsd_ubus_start(struct statsd_ubus *uub)
{
	int ret;
	MSG(INFO, "Connecting to ubus\n");

	ret = ubus_connect_ctx(&uub->ctx, NULL);
	if (ret) {
		MSG(ERROR, "Failed to connect to ubus: %s\n", ubus_strerror(ret));
		return -1;
	}

	ret = ubus_add_object(&uub->ctx, &statsd_ubus_obj);
	if (ret) {
		MSG(ERROR, "Failed to add ubus object: %s\n", ubus_strerror(ret));
		ubus_free(&uub->ctx);
		return -1;
	}

	ubus_add_uloop(&uub->ctx);

	return 0;
}
