package main

import (
	"encoding/json"
	"log"
	"net/http"
	"os"
	"os/exec"
	"strconv"
	"time"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promhttp"
)

// source: https://gist.github.com/fl64/a86b3d375deb947fa11099dd374660da

// Structs für die verschiedenen UQMI Outputs
type PLMNInfo struct {
	Mode string `json:"mode"`
}

type SignalInfo struct {
	Type string  `json:"type"`
	RSSI int     `json:"rssi"`
	RSRQ int     `json:"rsrq"`
	RSRP int     `json:"rsrp"`
	SNR  float64 `json:"snr"`
}

type RadioInfo struct {
	ServiceStatus           string `json:"service_status"`
	TrueServiceStatus       string `json:"true_service_status"`
	PreferredDataPath       bool   `json:"preferred_data_path"`
	Domain                  string `json:"domain,omitempty"`
	Service                 string `json:"service,omitempty"`
	RoamingStatus           string `json:"roaming_status,omitempty"`
	Forbidden               bool   `json:"forbidden,omitempty"`
	MCC                     string `json:"mcc,omitempty"`
	MNC                     string `json:"mnc,omitempty"`
	TrackingAreaCode        int    `json:"tracking_area_code,omitempty"`
	EnodebID                int    `json:"enodeb_id,omitempty"`
	CellID                  int    `json:"cell_id,omitempty"`
	VoiceSupport            bool   `json:"voice_support,omitempty"`
	IMSVoiceSupport         bool   `json:"ims_voice_support,omitempty"`
	CellAccessStatus        string `json:"cell_access_status,omitempty"`
	RegistrationRestriction int    `json:"registration_restriction,omitempty"`
	RegistrationDomain      int    `json:"registration_domain,omitempty"`
	FiveGNSAAvailable       bool   `json:"5g_nsa_available,omitempty"`
	DCNRRestriction         bool   `json:"dcnr_restriction,omitempty"`
}

type ServingSystem struct {
	Registration    string   `json:"registration"`
	RadioInterface  []string `json:"radio_interface"`
	PlmnMCC         int      `json:"plmn_mcc"`
	PlmnMNC         int      `json:"plmn_mnc"`
	PlmnDescription string   `json:"plmn_description"`
	Roaming         bool     `json:"roaming"`
}

type SystemInfo struct {
	LTE     RadioInfo `json:"lte"`
	FiveGNR RadioInfo `json:"5gnr"`
}

// Define a struct for you collector that contains pointers
// to prometheus descriptors for each metric you wish to expose.
// Note you can also include fields of other types if they provide utility
// but we just won't be exposing them as metrics.
type UQMICollector struct {
	// device = modem = /dev/cdc-wdm0
	device string

	cellID            *prometheus.Desc
	cellInfo          *prometheus.Desc
	enodebID          *prometheus.Desc
	fiveGNSAAvailable *prometheus.Desc
	forbidden         *prometheus.Desc
	mcc               *prometheus.Desc
	mnc               *prometheus.Desc
	plmnInfo          *prometheus.Desc
	plmnMCC           *prometheus.Desc
	plmnMNC           *prometheus.Desc
	plmnRoaming       *prometheus.Desc
	preferredDataPath *prometheus.Desc
	roamingStatus     *prometheus.Desc
	serviceStatus     *prometheus.Desc
	signalRSSI        *prometheus.Desc
	signalRSRQ        *prometheus.Desc
	signalRSRP        *prometheus.Desc
	signalSNR         *prometheus.Desc
	trackingAreaCode  *prometheus.Desc
	trueServiceStatus *prometheus.Desc

	scrapeDuration *prometheus.Desc
	scrapeSuccess  *prometheus.Desc
}

// You must create a constructor for you collector that
// initializes every descriptor and returns a pointer to the collector
func NewUQMICollector(device string) prometheus.Collector {
	return &UQMICollector{
		device: device,

		signalRSSI: prometheus.NewDesc(
			"uqmi_signal_rssi_dbm",
			"Signal RSSI in dBm",
			[]string{"type"}, nil,
		),
		signalRSRQ: prometheus.NewDesc(
			"uqmi_signal_rsrq_db",
			"Signal RSRQ in dB",
			[]string{"type"}, nil,
		),
		signalRSRP: prometheus.NewDesc(
			"uqmi_signal_rsrp_dbm",
			"Signal RSRP in dBm",
			[]string{"type"}, nil,
		),
		signalSNR: prometheus.NewDesc(
			"uqmi_signal_snr_db",
			"Signal SNR in dB",
			[]string{"type"}, nil,
		),
		serviceStatus: prometheus.NewDesc(
			"uqmi_service_available",
			"Service availability",
			[]string{"type"}, nil,
		),
		roamingStatus: prometheus.NewDesc(
			"uqmi_roaming_status",
			"Roaming Status",
			[]string{"type"}, nil,
		),
		trueServiceStatus: prometheus.NewDesc(
			"uqmi_true_service_available",
			"True Service availability",
			[]string{"type"}, nil,
		),
		plmnInfo: prometheus.NewDesc(
			"uqmi_plmn_info",
			"Public Land Mobile Network information",
			[]string{"type", "registered", "description", "mode"}, nil,
		),
		plmnMCC: prometheus.NewDesc(
			"uqmi_plmn_mcc",
			"Public Land Mobile Network MCC",
			[]string{"type"}, nil,
		),
		plmnMNC: prometheus.NewDesc(
			"uqmi_plmn_mnc",
			"Public Land Mobile Network MNC",
			[]string{"type"}, nil,
		),
		plmnRoaming: prometheus.NewDesc(
			"uqmi_plmn_roaming",
			"Public Land Mobile Network Roaming",
			[]string{"type"}, nil,
		),
		preferredDataPath: prometheus.NewDesc(
			"uqmi_preferred_data_path",
			"Preferred Data Path",
			[]string{"type"}, nil,
		),
		forbidden: prometheus.NewDesc(
			"uqmi_forbidden",
			"Forbidden",
			[]string{"type"}, nil,
		),
		trackingAreaCode: prometheus.NewDesc(
			"uqmi_tracking_area_code",
			"Tracking Area Code",
			[]string{"type"}, nil,
		),
		enodebID: prometheus.NewDesc(
			"uqmi_enodeb_id",
			"eNodeB ID",
			[]string{"type"}, nil,
		),
		cellID: prometheus.NewDesc(
			"uqmi_cell_id",
			"Cell ID",
			[]string{"type"}, nil,
		),
		fiveGNSAAvailable: prometheus.NewDesc(
			"uqmi_fiveg_nsa_available",
			"5G NSA Available",
			[]string{"type"}, nil,
		),
		mcc: prometheus.NewDesc(
			"uqmi_mcc",
			"Mobile Country Code",
			[]string{"type"}, nil,
		),
		mnc: prometheus.NewDesc(
			"uqmi_mnc",
			"Mobilfunknetzkennzahl",
			[]string{"type"}, nil,
		),

		cellInfo: prometheus.NewDesc(
			"uqmi_cell_info",
			"Radio Cell information",
			[]string{"type", "domain", "service", "voice_support", "ims_voice_support",
				"cell_access_status", "registration_restriction",
				"registriction_domain", "dcnr_restriction"}, nil,
		),

		scrapeDuration: prometheus.NewDesc(
			"uqmi_scrape_duration_seconds",
			"Time spent scraping UQMI",
			nil, nil,
		),
		scrapeSuccess: prometheus.NewDesc(
			"uqmi_scrape_success",
			"Whether UQMI scrape was successful",
			nil, nil,
		),
	}
}

// Each and every collector must implement the Describe function.
// It essentially writes all descriptors to the prometheus desc channel.
func (c *UQMICollector) Describe(ch chan<- *prometheus.Desc) {
	ch <- c.cellID
	ch <- c.cellInfo
	ch <- c.enodebID
	ch <- c.fiveGNSAAvailable
	ch <- c.forbidden
	ch <- c.mcc
	ch <- c.mnc
	ch <- c.plmnInfo
	ch <- c.plmnMCC
	ch <- c.plmnMNC
	ch <- c.plmnRoaming
	ch <- c.preferredDataPath
	ch <- c.roamingStatus
	ch <- c.serviceStatus
	ch <- c.signalRSSI
	ch <- c.signalRSRQ
	ch <- c.signalRSRP
	ch <- c.signalSNR
	ch <- c.trackingAreaCode
	ch <- c.trueServiceStatus

	ch <- c.scrapeDuration
	ch <- c.scrapeSuccess
}

func (c *UQMICollector) Collect(ch chan<- prometheus.Metric) {
	start := time.Now()

	success := 1.0

	// PLMN
	var plmnMode string
	if plmn, err := c.getPLMNInfo(); err != nil {
		log.Printf("Failed to get PLMN info: %v", err)
		success = 0.0
	} else {
		plmnMode = plmn.Mode
	}

	if servingSystem, err := c.getServingSystem(); err != nil {
		log.Printf("failed to get serving system info %v", err)
		success = 0.0
	} else {
		// plmn_info
		ch <- prometheus.MustNewConstMetric(
			c.plmnInfo, prometheus.GaugeValue, 1,
			servingSystem.RadioInterface[0],
			servingSystem.Registration,
			servingSystem.PlmnDescription,
			plmnMode,
		)
		// plmn_mcc
		ch <- prometheus.MustNewConstMetric(
			c.plmnMCC, prometheus.GaugeValue, float64(servingSystem.PlmnMCC),
			servingSystem.RadioInterface[0],
		)

		// plmn_mnc
		ch <- prometheus.MustNewConstMetric(
			c.plmnMNC, prometheus.GaugeValue, float64(servingSystem.PlmnMNC),
			servingSystem.RadioInterface[0],
		)

		// plmn_roaming
		ch <- prometheus.MustNewConstMetric(
			c.plmnRoaming, prometheus.GaugeValue, Bool2float64(servingSystem.Roaming),
			servingSystem.RadioInterface[0],
		)

	}

	// read every radio from system-info
	if system, err := c.getSystemInfo(); err != nil {
		log.Printf("Failed to get system info: %v", err)
		success = 0.0
	} else {
		radios := map[string]RadioInfo{
			"lte":  system.LTE,
			"5gnr": system.FiveGNR,
		}

		for radioType, radio := range radios {
			// cell_id
			cell_id := radio.CellID
			ch <- prometheus.MustNewConstMetric(
				c.cellID, prometheus.GaugeValue, float64(cell_id),
				radioType,
			)
			// cell_info
			ch <- prometheus.MustNewConstMetric(
				c.cellInfo, prometheus.GaugeValue, 1,
				radioType,
				radio.Domain,
				radio.Service,
				strconv.FormatBool(radio.VoiceSupport),
				strconv.FormatBool(radio.IMSVoiceSupport),
				radio.CellAccessStatus,
				strconv.Itoa(radio.RegistrationRestriction),
				strconv.Itoa(radio.RegistrationDomain),
				strconv.FormatBool(radio.DCNRRestriction),
			)

			// enodeb_id
			ch <- prometheus.MustNewConstMetric(
				c.enodebID, prometheus.GaugeValue, float64(radio.EnodebID),
				radioType,
			)

			// fivegnsa_available
			ch <- prometheus.MustNewConstMetric(
				c.fiveGNSAAvailable, prometheus.GaugeValue, Bool2float64(radio.FiveGNSAAvailable),
				radioType,
			)
			// forbidden
			ch <- prometheus.MustNewConstMetric(
				c.forbidden, prometheus.GaugeValue, Bool2float64(radio.Forbidden),
				radioType,
			)

			// mcc
			var mcc float64
			if s, err := strconv.ParseFloat(radio.MCC, 64); err == nil {
				mcc = s
			}
			ch <- prometheus.MustNewConstMetric(
				c.mcc, prometheus.GaugeValue, mcc,
				radioType,
			)

			// mnc
			var mnc float64
			if s, err := strconv.ParseFloat(radio.MNC, 64); err == nil {
				mnc = s
			}
			ch <- prometheus.MustNewConstMetric(
				c.mnc, prometheus.GaugeValue, mnc,
				radioType,
			)

			// preffered_data_path
			ch <- prometheus.MustNewConstMetric(
				c.preferredDataPath, prometheus.GaugeValue, Bool2float64(radio.PreferredDataPath),
				radioType,
			)

			// roaming_status
			var roaming float64
			if radio.RoamingStatus == "off" {
				roaming = 0
			} else {
				roaming = 1
			}

			ch <- prometheus.MustNewConstMetric(
				c.roamingStatus, prometheus.GaugeValue, roaming,
				radioType,
			)

			// tracking_area_code
			ch <- prometheus.MustNewConstMetric(
				c.trackingAreaCode, prometheus.GaugeValue, float64(radio.TrackingAreaCode),
				radioType,
			)

			// service_status and true_service_status
			var status float64
			if radio.ServiceStatus == "available" {
				status = 1
			} else {
				status = 0
			}
			ch <- prometheus.MustNewConstMetric(
				c.serviceStatus, prometheus.GaugeValue, status,
				radioType,
			)
			if radio.TrueServiceStatus == "available" {
				status = 1
			} else {
				status = 0
			}
			ch <- prometheus.MustNewConstMetric(
				c.trueServiceStatus, prometheus.GaugeValue, status,
				radioType,
			)
		}

	}

	// Signal
	if signal, err := c.getSignalInfo(); err != nil {
		log.Printf("Failed to get signal info: %v", err)
		success = 0.0
	} else {

		ch <- prometheus.MustNewConstMetric(
			c.signalRSSI, prometheus.GaugeValue, float64(signal.RSSI),
			signal.Type,
		)

		ch <- prometheus.MustNewConstMetric(
			c.signalRSRQ, prometheus.GaugeValue, float64(signal.RSRQ),
			signal.Type,
		)
		ch <- prometheus.MustNewConstMetric(
			c.signalRSRP, prometheus.GaugeValue, float64(signal.RSRP),
			signal.Type,
		)
		ch <- prometheus.MustNewConstMetric(
			c.signalSNR, prometheus.GaugeValue, signal.SNR,
			signal.Type,
		)
	}

	duration := time.Since(start).Seconds()
	ch <- prometheus.MustNewConstMetric(c.scrapeDuration, prometheus.GaugeValue, duration)
	ch <- prometheus.MustNewConstMetric(c.scrapeSuccess, prometheus.GaugeValue, success)
}

// helper functions to get uqmi infos
func (c *UQMICollector) runUQMICommand(args ...string) ([]byte, error) {
	lock()
	fullArgs := append([]string{"-d", c.device}, args...)
	cmd := exec.Command("uqmi", fullArgs...)
	unlock()
	return cmd.Output()
}

func (c *UQMICollector) getPLMNInfo() (*PLMNInfo, error) {
	output, err := c.runUQMICommand("--get-plmn")
	if err != nil {
		return nil, err
	}

	var plmn PLMNInfo
	err = json.Unmarshal(output, &plmn)
	return &plmn, err
}

func (c *UQMICollector) getSignalInfo() (*SignalInfo, error) {
	output, err := c.runUQMICommand("--get-signal-info")
	if err != nil {
		return nil, err
	}

	var signal SignalInfo

	err = json.Unmarshal(output, &signal)
	return &signal, err
}

func (c *UQMICollector) getSystemInfo() (*SystemInfo, error) {
	output, err := c.runUQMICommand("--get-system-info")
	if err != nil {
		return nil, err
	}

	var system SystemInfo
	err = json.Unmarshal(output, &system)
	return &system, err
}

func (c *UQMICollector) getServingSystem() (*ServingSystem, error) {
	output, err := c.runUQMICommand("--get-serving-system")
	if err != nil {
		return nil, err
	}

	var system ServingSystem
	err = json.Unmarshal(output, &system)
	return &system, err
}

// other helper function
func Bool2float64(b bool) float64 {
	var i float64
	if b {
		i = 1
	} else {
		i = 0
	}
	return i
}

// lock mechanism because only one process at a time can use uqmi
const lockDir = "/tmp/uqmi.lock"

func lock() {
	// try until unlocked
	for {
		err := os.Mkdir(lockDir, 0755)
		if err == nil {
			return
		}
		// maybe add timeout ଘ(੭*ˊᵕˋ)੭* ੈ♡‧₊˚
		time.Sleep(1 * time.Second)
	}
}

func unlock() {
	os.Remove(lockDir)
}

func main() {
	uqmiCollector := NewUQMICollector("/dev/cdc-wdm0")

	promReg := prometheus.NewRegistry()
	promReg.MustRegister(uqmiCollector)

	handler := promhttp.HandlerFor(
		promReg,
		promhttp.HandlerOpts{
			EnableOpenMetrics: false,
		})

	http.Handle("/metrics", handler)
	log.Fatal(http.ListenAndServe(":9101", nil))
}
